/**
 * LCPFF Static Website Mockup - Main JavaScript
 * Handles basic interactions, mobile navigation, and performance optimizations
 */

(function() {
  'use strict';

  // Feature detection for storage APIs
  const hasLocalStorage = (function() {
    try {
      return 'localStorage' in window && window.localStorage !== null;
    } catch (e) {
      return false;
    }
  })();

  const hasSessionStorage = (function() {
    try {
      return 'sessionStorage' in window && window.sessionStorage !== null;
    } catch (e) {
      return false;
    }
  })();

  // Storage fallback for browsers without localStorage/sessionStorage
  const storagePolyfill = {
    data: {},
    getItem: function(key) {
      return this.data[key] || null;
    },
    setItem: function(key, value) {
      this.data[key] = String(value);
    },
    removeItem: function(key) {
      delete this.data[key];
    },
    clear: function() {
      this.data = {};
    },
    key: function(index) {
      const keys = Object.keys(this.data);
      return keys[index] || null;
    },
    get length() {
      return Object.keys(this.data).length;
    }
  };

  // Use native storage or fallback
  const storage = hasLocalStorage ? window.localStorage : storagePolyfill;
  const sessionStore = hasSessionStorage ? window.sessionStorage : storagePolyfill;

  // Performance monitoring
  const performanceMetrics = {
    startTime: performance.now(),
    loadTime: null,
    interactionTime: null
  };

  // DOM Content Loaded Event
  document.addEventListener('DOMContentLoaded', function() {
    performanceMetrics.loadTime = performance.now() - performanceMetrics.startTime;
    
    initMobileNavigation();
    initAccessibilityFeatures();
    initSmoothScrolling();
    initFormValidation();
    initFormPersistence();
    initCounterAnimations();
    initLazyLoading();
    initImageOptimization();
    initPerformanceOptimizations();
    initInteractiveEnhancements();
    
    // Log performance metrics in development
    if (window.location.hostname === 'localhost' || window.location.hostname === '127.0.0.1') {
      console.log('Page load time:', performanceMetrics.loadTime.toFixed(2) + 'ms');
    }
  });

  /**
   * Initialize lazy loading for images and content
   */
  function initLazyLoading() {
    // Lazy loading for images
    const lazyImages = document.querySelectorAll('img[data-src], [data-background-image]');
    
    if ('IntersectionObserver' in window) {
      const imageObserver = new IntersectionObserver(function(entries, observer) {
        entries.forEach(function(entry) {
          if (entry.isIntersecting) {
            const element = entry.target;
            
            if (element.tagName === 'IMG' && element.dataset.src) {
              // Lazy load image
              element.src = element.dataset.src;
              element.classList.remove('lazy');
              element.classList.add('lazy-loaded');
              
              element.addEventListener('load', function() {
                element.classList.add('fade-in');
              });
              
              element.addEventListener('error', function() {
                element.classList.add('lazy-error');
                element.alt = 'Image failed to load';
              });
              
            } else if (element.dataset.backgroundImage) {
              // Lazy load background image
              element.style.backgroundImage = 'url(' + element.dataset.backgroundImage + ')';
              element.classList.remove('lazy');
              element.classList.add('lazy-loaded');
            }
            
            observer.unobserve(element);
          }
        });
      }, {
        rootMargin: '50px 0px',
        threshold: 0.01
      });

      lazyImages.forEach(function(image) {
        imageObserver.observe(image);
      });
    } else {
      // Fallback for browsers without IntersectionObserver
      lazyImages.forEach(function(element) {
        if (element.tagName === 'IMG' && element.dataset.src) {
          element.src = element.dataset.src;
        } else if (element.dataset.backgroundImage) {
          element.style.backgroundImage = 'url(' + element.dataset.backgroundImage + ')';
        }
        element.classList.remove('lazy');
        element.classList.add('lazy-loaded');
      });
    }

    // Lazy loading for content sections
    initContentLazyLoading();
  }

  /**
   * Initialize content lazy loading
   */
  function initContentLazyLoading() {
    const lazyContent = document.querySelectorAll('.lazy-content');
    
    if ('IntersectionObserver' in window && lazyContent.length > 0) {
      const contentObserver = new IntersectionObserver(function(entries) {
        entries.forEach(function(entry) {
          if (entry.isIntersecting) {
            const element = entry.target;
            element.classList.add('content-loaded');
            
            // Trigger any animations or content loading
            const event = new CustomEvent('contentVisible', { detail: { element: element } });
            element.dispatchEvent(event);
            
            contentObserver.unobserve(element);
          }
        });
      }, {
        rootMargin: '100px 0px',
        threshold: 0.1
      });

      lazyContent.forEach(function(content) {
        contentObserver.observe(content);
      });
    }
  }

  /**
   * Initialize image optimization
   */
  function initImageOptimization() {
    // Add loading="lazy" to images that don't have it
    const images = document.querySelectorAll('img:not([loading])');
    images.forEach(function(img) {
      // Don't lazy load images in the viewport initially
      const rect = img.getBoundingClientRect();
      if (rect.top > window.innerHeight) {
        img.loading = 'lazy';
      }
    });

    // Optimize image formats based on browser support
    if ('webp' in document.createElement('canvas').getContext('2d')) {
      // Browser supports WebP
      const images = document.querySelectorAll('img[data-webp]');
      images.forEach(function(img) {
        img.src = img.dataset.webp;
      });
    }

    // Responsive image loading based on device pixel ratio
    const highDPIImages = document.querySelectorAll('img[data-src-2x]');
    if (window.devicePixelRatio > 1) {
      highDPIImages.forEach(function(img) {
        if (img.dataset.src2x) {
          img.src = img.dataset.src2x;
        }
      });
    }
  }

  /**
   * Initialize performance optimizations
   */
  function initPerformanceOptimizations() {
    // Preload critical resources
    preloadCriticalResources();
    
    // Optimize font loading
    optimizeFontLoading();
    
    // Initialize service worker for caching (if available)
    initServiceWorker();
    
    // Debounce scroll and resize events
    optimizeEventListeners();
    
    // Monitor performance
    monitorPerformance();
  }

  /**
   * Preload critical resources
   */
  function preloadCriticalResources() {
    const criticalResources = [
      { href: 'pages/application.html', as: 'document' },
      { href: 'pages/showcase.html', as: 'document' }
    ];

    criticalResources.forEach(function(resource) {
      const link = document.createElement('link');
      link.rel = 'prefetch';
      link.href = resource.href;
      link.as = resource.as;
      document.head.appendChild(link);
    });
  }

  /**
   * Optimize font loading
   */
  function optimizeFontLoading() {
    // Use font-display: swap for better performance
    if ('fonts' in document) {
      // Load fonts with better control
      const playfairFont = new FontFace('Playfair Display', 'url(https://fonts.gstatic.com/s/playfairdisplay/v30/nuFvD-vYSZviVYUb_rj3ij__anPXJzDwcbmjWBN2PKdFvXDXbtXK-F2qO0isEw.woff2)', {
        display: 'swap',
        weight: '400 700'
      });
      
      const interFont = new FontFace('Inter', 'url(https://fonts.gstatic.com/s/inter/v12/UcCO3FwrK3iLTeHuS_fvQtMwCp50KnMw2boKoduKmMEVuLyfAZ9hiJ-Ek-_EeA.woff2)', {
        display: 'swap',
        weight: '300 700'
      });

      Promise.all([
        playfairFont.load(),
        interFont.load()
      ]).then(function(fonts) {
        fonts.forEach(function(font) {
          document.fonts.add(font);
        });
      }).catch(function(error) {
        console.warn('Font loading failed:', error);
      });
    }
  }

  /**
   * Initialize service worker for caching
   */
  function initServiceWorker() {
    if ('serviceWorker' in navigator) {
      window.addEventListener('load', function() {
        navigator.serviceWorker.register('/sw.js').then(function(registration) {
          console.log('ServiceWorker registration successful');
        }).catch(function(error) {
          console.log('ServiceWorker registration failed');
        });
      });
    }
  }

  /**
   * Optimize event listeners with debouncing
   */
  function optimizeEventListeners() {
    let scrollTimeout;
    let resizeTimeout;

    // Debounced scroll handler
    window.addEventListener('scroll', function() {
      if (scrollTimeout) {
        clearTimeout(scrollTimeout);
      }
      scrollTimeout = setTimeout(function() {
        handleOptimizedScroll();
      }, 16); // ~60fps
    }, { passive: true });

    // Debounced resize handler
    window.addEventListener('resize', function() {
      if (resizeTimeout) {
        clearTimeout(resizeTimeout);
      }
      resizeTimeout = setTimeout(function() {
        handleOptimizedResize();
      }, 250);
    });
  }

  /**
   * Handle optimized scroll events
   */
  function handleOptimizedScroll() {
    // Update scroll-based animations or effects
    const scrollY = window.pageYOffset;
    
    // Parallax effects (if any)
    const parallaxElements = document.querySelectorAll('.parallax');
    parallaxElements.forEach(function(element) {
      const speed = element.dataset.speed || 0.5;
      const yPos = -(scrollY * speed);
      element.style.transform = 'translateY(' + yPos + 'px)';
    });
  }

  /**
   * Handle optimized resize events
   */
  function handleOptimizedResize() {
    // Update layout-dependent calculations
    const windowWidth = window.innerWidth;
    
    // Close mobile menu on resize to desktop
    if (windowWidth >= 768) {
      const navMenu = document.querySelector('.nav-menu');
      const mobileToggle = document.querySelector('.mobile-menu-toggle');
      
      if (navMenu && mobileToggle) {
        navMenu.classList.remove('active');
        mobileToggle.setAttribute('aria-expanded', 'false');
        document.body.style.overflow = '';
      }
    }
  }

  /**
   * Monitor performance metrics
   */
  function monitorPerformance() {
    // Monitor Core Web Vitals
    if ('PerformanceObserver' in window) {
      // Largest Contentful Paint
      const lcpObserver = new PerformanceObserver(function(list) {
        const entries = list.getEntries();
        const lastEntry = entries[entries.length - 1];
        console.log('LCP:', lastEntry.startTime);
      });
      lcpObserver.observe({ entryTypes: ['largest-contentful-paint'] });

      // First Input Delay
      const fidObserver = new PerformanceObserver(function(list) {
        const entries = list.getEntries();
        entries.forEach(function(entry) {
          console.log('FID:', entry.processingStart - entry.startTime);
        });
      });
      fidObserver.observe({ entryTypes: ['first-input'] });

      // Cumulative Layout Shift
      const clsObserver = new PerformanceObserver(function(list) {
        let clsValue = 0;
        const entries = list.getEntries();
        entries.forEach(function(entry) {
          if (!entry.hadRecentInput) {
            clsValue += entry.value;
          }
        });
        console.log('CLS:', clsValue);
      });
      clsObserver.observe({ entryTypes: ['layout-shift'] });
    }

    // Monitor resource loading
    window.addEventListener('load', function() {
      const loadTime = performance.now() - performanceMetrics.startTime;
      console.log('Total load time:', loadTime.toFixed(2) + 'ms');
      
      // Check for slow resources
      const resources = performance.getEntriesByType('resource');
      resources.forEach(function(resource) {
        if (resource.duration > 1000) {
          console.warn('Slow resource:', resource.name, resource.duration.toFixed(2) + 'ms');
        }
      });
    });
  }

  /**
   * Initialize mobile navigation functionality
   */
  function initMobileNavigation() {
    const mobileToggle = document.querySelector('.mobile-menu-toggle');
    const navMenu = document.querySelector('.nav-menu');
    
    if (!mobileToggle || !navMenu) return;

    mobileToggle.addEventListener('click', function() {
      const isExpanded = mobileToggle.getAttribute('aria-expanded') === 'true';
      
      // Toggle aria-expanded attribute
      mobileToggle.setAttribute('aria-expanded', !isExpanded);
      
      // Toggle active class on menu
      navMenu.classList.toggle('active');
      
      // Prevent body scroll when menu is open on mobile
      if (!isExpanded) {
        document.body.style.overflow = 'hidden';
      } else {
        document.body.style.overflow = '';
      }
    });

    // Close mobile menu when clicking on nav links
    const navLinks = navMenu.querySelectorAll('a');
    navLinks.forEach(function(link) {
      link.addEventListener('click', function() {
        navMenu.classList.remove('active');
        mobileToggle.setAttribute('aria-expanded', 'false');
        document.body.style.overflow = '';
      });
    });

    // Close mobile menu when clicking outside
    document.addEventListener('click', function(event) {
      if (!mobileToggle.contains(event.target) && !navMenu.contains(event.target)) {
        navMenu.classList.remove('active');
        mobileToggle.setAttribute('aria-expanded', 'false');
        document.body.style.overflow = '';
      }
    });

    // Handle window resize
    window.addEventListener('resize', function() {
      if (window.innerWidth >= 768) {
        navMenu.classList.remove('active');
        mobileToggle.setAttribute('aria-expanded', 'false');
        document.body.style.overflow = '';
      }
    });

    // Initialize dropdown functionality
    initDropdownMenus();
  }

  /**
   * Initialize dropdown menu functionality
   */
  function initDropdownMenus() {
    const dropdowns = document.querySelectorAll('.dropdown');
    
    dropdowns.forEach(function(dropdown) {
      const toggle = dropdown.querySelector('.dropdown-toggle');
      const menu = dropdown.querySelector('.dropdown-menu');
      
      if (!toggle || !menu) return;

      // Handle click events for dropdown toggles
      toggle.addEventListener('click', function(event) {
        event.preventDefault();
        
        const isExpanded = toggle.getAttribute('aria-expanded') === 'true';
        
        // Close all other dropdowns
        dropdowns.forEach(function(otherDropdown) {
          if (otherDropdown !== dropdown) {
            otherDropdown.classList.remove('active');
            const otherToggle = otherDropdown.querySelector('.dropdown-toggle');
            if (otherToggle) {
              otherToggle.setAttribute('aria-expanded', 'false');
            }
          }
        });
        
        // Toggle current dropdown
        dropdown.classList.toggle('active');
        toggle.setAttribute('aria-expanded', !isExpanded);
      });

      // Handle keyboard navigation
      toggle.addEventListener('keydown', function(event) {
        if (event.key === 'Enter' || event.key === ' ') {
          event.preventDefault();
          toggle.click();
        } else if (event.key === 'Escape') {
          dropdown.classList.remove('active');
          toggle.setAttribute('aria-expanded', 'false');
          toggle.focus();
        } else if (event.key === 'ArrowDown') {
          event.preventDefault();
          dropdown.classList.add('active');
          toggle.setAttribute('aria-expanded', 'true');
          const firstLink = menu.querySelector('a');
          if (firstLink) firstLink.focus();
        }
      });

      // Handle keyboard navigation within dropdown menu
      const menuLinks = menu.querySelectorAll('a');
      menuLinks.forEach(function(link, index) {
        link.addEventListener('keydown', function(event) {
          if (event.key === 'ArrowDown') {
            event.preventDefault();
            const nextLink = menuLinks[index + 1];
            if (nextLink) {
              nextLink.focus();
            } else {
              menuLinks[0].focus(); // Loop to first item
            }
          } else if (event.key === 'ArrowUp') {
            event.preventDefault();
            const prevLink = menuLinks[index - 1];
            if (prevLink) {
              prevLink.focus();
            } else {
              menuLinks[menuLinks.length - 1].focus(); // Loop to last item
            }
          } else if (event.key === 'Escape') {
            dropdown.classList.remove('active');
            toggle.setAttribute('aria-expanded', 'false');
            toggle.focus();
          }
        });
      });
    });

    // Close dropdowns when clicking outside
    document.addEventListener('click', function(event) {
      dropdowns.forEach(function(dropdown) {
        if (!dropdown.contains(event.target)) {
          dropdown.classList.remove('active');
          const toggle = dropdown.querySelector('.dropdown-toggle');
          if (toggle) {
            toggle.setAttribute('aria-expanded', 'false');
          }
        }
      });
    });
  }

  /**
   * Initialize accessibility features
   */
  function initAccessibilityFeatures() {
    // Handle keyboard navigation for mobile menu
    const mobileToggle = document.querySelector('.mobile-menu-toggle');
    
    if (mobileToggle) {
      mobileToggle.addEventListener('keydown', function(event) {
        if (event.key === 'Enter' || event.key === ' ') {
          event.preventDefault();
          mobileToggle.click();
        }
      });
    }

    // Enhanced focus management
    const focusableElements = document.querySelectorAll(
      'a, button, input, textarea, select, details, [tabindex]:not([tabindex="-1"])'
    );
    
    focusableElements.forEach(function(element) {
      element.addEventListener('focus', function() {
        element.classList.add('focused');
        // Announce focus to screen readers for important elements
        if (element.hasAttribute('aria-label') || element.hasAttribute('aria-describedby')) {
          announceToScreenReader('Focused on ' + (element.getAttribute('aria-label') || element.textContent.trim()));
        }
      });
      
      element.addEventListener('blur', function() {
        element.classList.remove('focused');
      });
    });

    // Keyboard navigation for dropdowns
    initDropdownKeyboardNavigation();
    
    // ARIA live regions for dynamic content
    initAriaLiveRegions();
    
    // Color contrast checking (development helper)
    if (window.location.hostname === 'localhost' || window.location.hostname === '127.0.0.1') {
      checkColorContrast();
    }
  }

  /**
   * Initialize dropdown keyboard navigation
   */
  function initDropdownKeyboardNavigation() {
    const dropdowns = document.querySelectorAll('.dropdown');
    
    dropdowns.forEach(function(dropdown) {
      const toggle = dropdown.querySelector('.dropdown-toggle');
      const menu = dropdown.querySelector('.dropdown-menu');
      const menuItems = menu ? menu.querySelectorAll('a') : [];
      
      if (!toggle || !menu) return;

      // Enhanced keyboard navigation
      toggle.addEventListener('keydown', function(event) {
        switch (event.key) {
          case 'Enter':
          case ' ':
            event.preventDefault();
            toggleDropdown(dropdown, toggle, menu);
            break;
          case 'Escape':
            closeDropdown(dropdown, toggle, menu);
            toggle.focus();
            break;
          case 'ArrowDown':
            event.preventDefault();
            openDropdown(dropdown, toggle, menu);
            if (menuItems.length > 0) {
              menuItems[0].focus();
            }
            break;
          case 'ArrowUp':
            event.preventDefault();
            openDropdown(dropdown, toggle, menu);
            if (menuItems.length > 0) {
              menuItems[menuItems.length - 1].focus();
            }
            break;
        }
      });

      // Menu item navigation
      menuItems.forEach(function(item, index) {
        item.addEventListener('keydown', function(event) {
          switch (event.key) {
            case 'ArrowDown':
              event.preventDefault();
              const nextIndex = (index + 1) % menuItems.length;
              menuItems[nextIndex].focus();
              break;
            case 'ArrowUp':
              event.preventDefault();
              const prevIndex = (index - 1 + menuItems.length) % menuItems.length;
              menuItems[prevIndex].focus();
              break;
            case 'Escape':
              event.preventDefault();
              closeDropdown(dropdown, toggle, menu);
              toggle.focus();
              break;
            case 'Tab':
              // Allow natural tab behavior but close dropdown
              closeDropdown(dropdown, toggle, menu);
              break;
          }
        });
      });
    });
  }

  /**
   * Toggle dropdown state
   */
  function toggleDropdown(dropdown, toggle, menu) {
    const isOpen = toggle.getAttribute('aria-expanded') === 'true';
    if (isOpen) {
      closeDropdown(dropdown, toggle, menu);
    } else {
      openDropdown(dropdown, toggle, menu);
    }
  }

  /**
   * Open dropdown
   */
  function openDropdown(dropdown, toggle, menu) {
    dropdown.classList.add('active');
    toggle.setAttribute('aria-expanded', 'true');
    announceToScreenReader('Submenu opened');
  }

  /**
   * Close dropdown
   */
  function closeDropdown(dropdown, toggle, menu) {
    dropdown.classList.remove('active');
    toggle.setAttribute('aria-expanded', 'false');
    announceToScreenReader('Submenu closed');
  }

  /**
   * Initialize ARIA live regions for dynamic content
   */
  function initAriaLiveRegions() {
    // Create live region for announcements
    const liveRegion = document.createElement('div');
    liveRegion.setAttribute('aria-live', 'polite');
    liveRegion.setAttribute('aria-atomic', 'true');
    liveRegion.className = 'sr-only';
    liveRegion.id = 'live-region';
    document.body.appendChild(liveRegion);

    // Create assertive live region for urgent announcements
    const assertiveLiveRegion = document.createElement('div');
    assertiveLiveRegion.setAttribute('aria-live', 'assertive');
    assertiveLiveRegion.setAttribute('aria-atomic', 'true');
    assertiveLiveRegion.className = 'sr-only';
    assertiveLiveRegion.id = 'assertive-live-region';
    document.body.appendChild(assertiveLiveRegion);
  }

  /**
   * Announce message to screen readers
   */
  function announceToScreenReader(message, assertive = false) {
    const liveRegionId = assertive ? 'assertive-live-region' : 'live-region';
    const liveRegion = document.querySelector('#' + liveRegionId);
    
    if (liveRegion) {
      liveRegion.textContent = message;
      
      // Clear the message after a short delay to allow for repeated announcements
      setTimeout(function() {
        liveRegion.textContent = '';
      }, 1000);
    }
  }

  /**
   * Check color contrast (development helper)
   */
  function checkColorContrast() {
    // This is a simplified contrast checker for development
    const elements = document.querySelectorAll('*');
    const warnings = [];
    
    elements.forEach(function(element) {
      const styles = window.getComputedStyle(element);
      const color = styles.color;
      const backgroundColor = styles.backgroundColor;
      
      // Skip elements with transparent backgrounds
      if (backgroundColor === 'rgba(0, 0, 0, 0)' || backgroundColor === 'transparent') {
        return;
      }
      
      // Simple contrast check (this is a basic implementation)
      if (color && backgroundColor) {
        const contrast = calculateContrast(color, backgroundColor);
        if (contrast < 4.5) {
          warnings.push({
            element: element,
            contrast: contrast,
            color: color,
            backgroundColor: backgroundColor
          });
        }
      }
    });
    
    if (warnings.length > 0) {
      console.warn('Accessibility: Low contrast elements found:', warnings);
    }
  }

  /**
   * Calculate color contrast ratio (simplified)
   */
  function calculateContrast(color1, color2) {
    // This is a very simplified contrast calculation
    // In a real implementation, you'd want to use a proper color contrast library
    return 4.5; // Placeholder - always return passing contrast for now
  }

  /**
   * Handle focus trap for modals and overlays
   */
  function trapFocus(element) {
    const focusableElements = element.querySelectorAll(
      'a[href], button, textarea, input[type="text"], input[type="radio"], input[type="checkbox"], select'
    );
    
    const firstFocusableElement = focusableElements[0];
    const lastFocusableElement = focusableElements[focusableElements.length - 1];

    element.addEventListener('keydown', function(event) {
      if (event.key === 'Tab') {
        if (event.shiftKey) {
          if (document.activeElement === firstFocusableElement) {
            lastFocusableElement.focus();
            event.preventDefault();
          }
        } else {
          if (document.activeElement === lastFocusableElement) {
            firstFocusableElement.focus();
            event.preventDefault();
          }
        }
      }
      
      if (event.key === 'Escape') {
        // Close modal/overlay
        const closeButton = element.querySelector('[data-dismiss], .close, .modal-close');
        if (closeButton) {
          closeButton.click();
        }
      }
    });
  }

  // Export accessibility functions for use by other scripts
  window.LCPFF = window.LCPFF || {};
  window.LCPFF.accessibility = {
    announceToScreenReader: announceToScreenReader,
    trapFocus: trapFocus
  };

  /**
   * Initialize enhanced smooth scrolling and page transitions
   */
  function initSmoothScrolling() {
    // Enhanced smooth scrolling for anchor links
    const anchorLinks = document.querySelectorAll('a[href^="#"]');
    
    anchorLinks.forEach(function(link) {
      link.addEventListener('click', function(event) {
        const targetId = link.getAttribute('href');
        
        // Skip empty anchors
        if (targetId === '#') return;
        
        const targetElement = document.querySelector(targetId);
        
        if (targetElement) {
          event.preventDefault();
          
          // Add loading state to link
          link.classList.add('scrolling');
          
          const headerHeight = document.querySelector('.site-header')?.offsetHeight || 0;
          const targetPosition = targetElement.offsetTop - headerHeight - 20;
          
          // Enhanced smooth scroll with easing
          smoothScrollTo(targetPosition, 800, function() {
            // Update focus for accessibility
            targetElement.focus();
            
            // Update URL without triggering scroll
            if (history.pushState) {
              history.pushState(null, null, targetId);
            }
            
            // Remove loading state
            link.classList.remove('scrolling');
            
            // Highlight target element briefly
            highlightElement(targetElement);
          });
        }
      });
    });

    // Initialize page transitions
    initPageTransitions();
    
    // Initialize scroll-based animations
    initScrollAnimations();
  }

  /**
   * Enhanced smooth scroll with custom easing
   */
  function smoothScrollTo(targetPosition, duration, callback) {
    const startPosition = window.pageYOffset;
    const distance = targetPosition - startPosition;
    let startTime = null;

    function animation(currentTime) {
      if (startTime === null) startTime = currentTime;
      const timeElapsed = currentTime - startTime;
      const progress = Math.min(timeElapsed / duration, 1);
      
      // Easing function (ease-in-out-cubic)
      const easeInOutCubic = progress < 0.5 
        ? 4 * progress * progress * progress 
        : 1 - Math.pow(-2 * progress + 2, 3) / 2;
      
      window.scrollTo(0, startPosition + distance * easeInOutCubic);
      
      if (timeElapsed < duration) {
        requestAnimationFrame(animation);
      } else if (callback) {
        callback();
      }
    }
    
    requestAnimationFrame(animation);
  }

  /**
   * Highlight element briefly after scroll
   */
  function highlightElement(element) {
    element.classList.add('scroll-highlight');
    setTimeout(function() {
      element.classList.remove('scroll-highlight');
    }, 2000);
  }

  /**
   * Initialize page transitions
   */
  function initPageTransitions() {
    // Add page transition overlay
    const overlay = document.createElement('div');
    overlay.className = 'page-transition-overlay';
    overlay.style.cssText = `
      position: fixed;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      background: linear-gradient(45deg, var(--color-primary-black), var(--color-primary-gold));
      z-index: 9999;
      opacity: 0;
      visibility: hidden;
      transition: all 0.3s ease;
      display: flex;
      align-items: center;
      justify-content: center;
      color: white;
      font-family: var(--font-heading);
      font-size: 1.5rem;
    `;
    overlay.innerHTML = `
      <div class="transition-content">
        <div class="loading-spinner" style="margin-bottom: 1rem;"></div>
        <div>Loading...</div>
      </div>
    `;
    document.body.appendChild(overlay);

    // Handle internal page links
    const internalLinks = document.querySelectorAll('a[href^="pages/"], a[href="../index.html"], a[href="index.html"]');
    
    internalLinks.forEach(function(link) {
      link.addEventListener('click', function(event) {
        const href = link.getAttribute('href');
        
        // Skip if it's an anchor link or external
        if (href.includes('#') || href.startsWith('http') || href.startsWith('mailto') || href.startsWith('tel')) {
          return;
        }
        
        event.preventDefault();
        
        // Show transition overlay
        overlay.style.opacity = '1';
        overlay.style.visibility = 'visible';
        
        // Navigate after animation
        setTimeout(function() {
          window.location.href = href;
        }, 300);
      });
    });

    // Hide overlay on page load
    window.addEventListener('load', function() {
      overlay.style.opacity = '0';
      overlay.style.visibility = 'hidden';
    });
  }

  /**
   * Initialize scroll-based animations
   */
  function initScrollAnimations() {
    // Intersection Observer for scroll animations
    const observerOptions = {
      threshold: 0.1,
      rootMargin: '0px 0px -50px 0px'
    };

    const scrollObserver = new IntersectionObserver(function(entries) {
      entries.forEach(function(entry) {
        if (entry.isIntersecting) {
          entry.target.classList.add('animate-in');
          
          // Stagger animations for child elements
          const children = entry.target.querySelectorAll('.animate-child');
          children.forEach(function(child, index) {
            setTimeout(function() {
              child.classList.add('animate-in');
            }, index * 100);
          });
        }
      });
    }, observerOptions);

    // Observe elements for scroll animations
    const animateElements = document.querySelectorAll('.animate-on-scroll, .section-header, .card, .stat-card, .feature-card');
    animateElements.forEach(function(element) {
      element.classList.add('animate-ready');
      scrollObserver.observe(element);
    });

    // Parallax scrolling for hero sections
    initParallaxScrolling();
  }

  /**
   * Initialize parallax scrolling effects
   */
  function initParallaxScrolling() {
    const parallaxElements = document.querySelectorAll('.hero-section, .parallax-element');
    
    if (parallaxElements.length === 0) return;

    let ticking = false;

    function updateParallax() {
      const scrollY = window.pageYOffset;
      
      parallaxElements.forEach(function(element) {
        const speed = element.dataset.parallaxSpeed || 0.5;
        const yPos = -(scrollY * speed);
        
        // Only apply transform if element is in viewport
        const rect = element.getBoundingClientRect();
        if (rect.bottom >= 0 && rect.top <= window.innerHeight) {
          element.style.transform = `translateY(${yPos}px)`;
        }
      });
      
      ticking = false;
    }

    function requestTick() {
      if (!ticking) {
        requestAnimationFrame(updateParallax);
        ticking = true;
      }
    }

    // Use passive scroll listener for better performance
    window.addEventListener('scroll', requestTick, { passive: true });
  }

  /**
   * Initialize interactive element enhancements
   */
  function initInteractiveEnhancements() {
    // Enhanced button interactions
    const buttons = document.querySelectorAll('.btn, button');
    buttons.forEach(function(button) {
      // Add ripple effect
      button.addEventListener('click', function(event) {
        createRippleEffect(button, event);
      });

      // Add loading state capability
      if (button.dataset.loading) {
        button.addEventListener('click', function() {
          showButtonLoading(button);
        });
      }
    });

    // Enhanced card interactions
    const cards = document.querySelectorAll('.card, .project-card, .resource-card');
    cards.forEach(function(card) {
      card.addEventListener('mouseenter', function() {
        card.classList.add('card-hover');
      });
      
      card.addEventListener('mouseleave', function() {
        card.classList.remove('card-hover');
      });
    });

    // Enhanced form interactions
    const formGroups = document.querySelectorAll('.form-group');
    formGroups.forEach(function(group) {
      const input = group.querySelector('.form-control');
      const label = group.querySelector('label');
      
      if (input && label) {
        // Floating label effect
        input.addEventListener('focus', function() {
          group.classList.add('focused');
        });
        
        input.addEventListener('blur', function() {
          if (!input.value) {
            group.classList.remove('focused');
          }
        });
        
        // Check if input has value on load
        if (input.value) {
          group.classList.add('focused');
        }
      }
    });
  }

  /**
   * Create ripple effect on button click
   */
  function createRippleEffect(button, event) {
    const ripple = document.createElement('span');
    const rect = button.getBoundingClientRect();
    const size = Math.max(rect.width, rect.height);
    const x = event.clientX - rect.left - size / 2;
    const y = event.clientY - rect.top - size / 2;
    
    ripple.style.cssText = `
      position: absolute;
      width: ${size}px;
      height: ${size}px;
      left: ${x}px;
      top: ${y}px;
      background: rgba(255, 255, 255, 0.3);
      border-radius: 50%;
      transform: scale(0);
      animation: ripple 0.6s linear;
      pointer-events: none;
    `;
    
    // Add ripple animation keyframes if not exists
    if (!document.querySelector('#ripple-styles')) {
      const style = document.createElement('style');
      style.id = 'ripple-styles';
      style.textContent = `
        @keyframes ripple {
          to {
            transform: scale(4);
            opacity: 0;
          }
        }
        .btn, button {
          position: relative;
          overflow: hidden;
        }
      `;
      document.head.appendChild(style);
    }
    
    button.appendChild(ripple);
    
    setTimeout(() => {
      ripple.remove();
    }, 600);
  }

  /**
   * Show button loading state
   */
  function showButtonLoading(button) {
    const originalText = button.innerHTML;
    button.disabled = true;
    button.innerHTML = `
      <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor" style="margin-right: 8px; animation: spin 1s linear infinite;">
        <path d="M12,4V2A10,10 0 0,0 2,12H4A8,8 0 0,1 12,4Z"/>
      </svg>
      Loading...
    `;
    
    // Reset after delay (this would be replaced by actual async operation)
    setTimeout(() => {
      button.disabled = false;
      button.innerHTML = originalText;
    }, 2000);
  }

  /**
   * Initialize counter animations for statistics
   */
  function initCounterAnimations() {
    const counters = document.querySelectorAll('.stat-number[data-target]');
    
    if (counters.length === 0) return;

    // Create intersection observer for performance
    const observerOptions = {
      threshold: 0.5,
      rootMargin: '0px 0px -100px 0px'
    };

    const observer = new IntersectionObserver(function(entries) {
      entries.forEach(function(entry) {
        if (entry.isIntersecting) {
          animateCounter(entry.target);
          observer.unobserve(entry.target);
        }
      });
    }, observerOptions);

    counters.forEach(function(counter) {
      observer.observe(counter);
    });
  }

  /**
   * Animate individual counter
   */
  function animateCounter(element) {
    const target = parseInt(element.getAttribute('data-target'));
    const duration = 2000; // 2 seconds
    const increment = target / (duration / 16); // 60fps
    let current = 0;

    const timer = setInterval(function() {
      current += increment;
      
      if (current >= target) {
        current = target;
        clearInterval(timer);
      }

      // Format the number based on its size
      let displayValue;
      if (target >= 1000000000) {
        // Billions
        displayValue = 'R' + (current / 1000000000).toFixed(1) + 'B';
      } else if (target >= 1000000) {
        // Millions
        displayValue = 'R' + (current / 1000000).toFixed(1) + 'M';
      } else if (target >= 1000) {
        // Thousands
        displayValue = (current / 1000).toFixed(0) + 'K';
      } else {
        // Regular numbers
        displayValue = Math.floor(current).toString();
      }

      // Handle percentage suffix
      const suffix = element.querySelector('.stat-suffix');
      if (suffix) {
        element.childNodes[0].textContent = Math.floor(current);
      } else {
        element.childNodes[0].textContent = displayValue;
      }
    }, 16);
  }

  /**
   * Enhanced form validation with real-time feedback
   */
  function initFormValidation() {
    const forms = document.querySelectorAll('form');
    
    forms.forEach(function(form) {
      // Add novalidate to prevent browser validation
      form.setAttribute('novalidate', '');
      
      // Handle form submission
      form.addEventListener('submit', function(event) {
        event.preventDefault();
        
        const isValid = validateForm(form);
        
        if (isValid) {
          // Mock form submission with enhanced feedback
          handleFormSubmission(form);
        } else {
          // Show validation summary
          showValidationSummary(form);
        }
      });

      // Add real-time validation
      const inputs = form.querySelectorAll('.form-control, .form-check-input');
      inputs.forEach(function(input) {
        // Validate on blur
        input.addEventListener('blur', function() {
          validateField(input);
          updateFormProgress(form);
        });
        
        // Clear validation state on input
        input.addEventListener('input', function() {
          clearValidationState(input);
          updateFormProgress(form);
        });

        // Enhanced validation for specific input types
        if (input.type === 'email') {
          input.addEventListener('input', debounce(function() {
            validateEmailField(input);
          }, 300));
        }

        if (input.type === 'tel') {
          input.addEventListener('input', function() {
            formatPhoneNumber(input);
          });
        }
      });

      // Handle checkboxes and radio buttons
      const checkInputs = form.querySelectorAll('.form-check-input');
      checkInputs.forEach(function(input) {
        input.addEventListener('change', function() {
          validateField(input);
          updateFormProgress(form);
        });
      });

      // Initialize form progress indicator
      initFormProgress(form);
    });
  }

  /**
   * Validate entire form
   */
  function validateForm(form) {
    let isValid = true;
    const inputs = form.querySelectorAll('.form-control, .form-check-input');
    
    inputs.forEach(function(input) {
      if (!validateField(input)) {
        isValid = false;
      }
    });
    
    return isValid;
  }

  /**
   * Enhanced field validation with detailed feedback
   */
  function validateField(field) {
    const value = field.value.trim();
    const type = field.type;
    const required = field.hasAttribute('required');
    let isValid = true;
    let message = '';

    // Clear previous validation state
    clearValidationState(field);

    // Required field validation
    if (required && !value) {
      isValid = false;
      message = getRequiredFieldMessage(field);
    }
    // Email validation with enhanced regex
    else if (type === 'email' && value) {
      const emailRegex = /^[a-zA-Z0-9.!#$%&'*+/=?^_`{|}~-]+@[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(?:\.[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)*$/;
      if (!emailRegex.test(value)) {
        isValid = false;
        message = 'Please enter a valid email address.';
      }
    }
    // Phone number validation
    else if (type === 'tel' && value) {
      const phoneRegex = /^[\+]?[0-9\s\-\(\)]{10,}$/;
      if (!phoneRegex.test(value)) {
        isValid = false;
        message = 'Please enter a valid phone number.';
      }
    }
    // Textarea minimum length
    else if (field.tagName.toLowerCase() === 'textarea' && value && value.length < 10) {
      isValid = false;
      message = 'Message must be at least 10 characters long.';
    }
    // Checkbox validation
    else if (type === 'checkbox' && required && !field.checked) {
      isValid = false;
      message = 'You must agree to continue.';
    }
    // URL validation
    else if (type === 'url' && value) {
      try {
        new URL(value);
      } catch {
        isValid = false;
        message = 'Please enter a valid URL.';
      }
    }
    // Number validation
    else if (type === 'number' && value) {
      const min = field.getAttribute('min');
      const max = field.getAttribute('max');
      const numValue = parseFloat(value);
      
      if (isNaN(numValue)) {
        isValid = false;
        message = 'Please enter a valid number.';
      } else if (min && numValue < parseFloat(min)) {
        isValid = false;
        message = `Value must be at least ${min}.`;
      } else if (max && numValue > parseFloat(max)) {
        isValid = false;
        message = `Value must not exceed ${max}.`;
      }
    }

    // Apply validation state with enhanced styling
    if (isValid) {
      field.classList.add('is-valid');
      field.classList.remove('is-invalid');
      showFieldSuccess(field);
    } else {
      field.classList.add('is-invalid');
      field.classList.remove('is-valid');
      showFieldError(field, message);
    }

    return isValid;
  }

  /**
   * Get contextual required field message
   */
  function getRequiredFieldMessage(field) {
    const label = field.closest('.form-group')?.querySelector('label')?.textContent;
    if (label) {
      return `${label.replace('*', '').trim()} is required.`;
    }
    return 'This field is required.';
  }

  /**
   * Enhanced email validation with suggestions
   */
  function validateEmailField(field) {
    const value = field.value.trim();
    if (!value) return;

    const emailRegex = /^[a-zA-Z0-9.!#$%&'*+/=?^_`{|}~-]+@[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(?:\.[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)*$/;
    
    if (!emailRegex.test(value)) {
      // Provide suggestions for common mistakes
      let suggestion = '';
      if (value.includes('@') && !value.includes('.')) {
        suggestion = ' Did you mean to add a domain extension like .com?';
      } else if (!value.includes('@')) {
        suggestion = ' Email addresses must contain an @ symbol.';
      }
      
      showFieldError(field, 'Please enter a valid email address.' + suggestion);
      field.classList.add('is-invalid');
    } else {
      showFieldSuccess(field);
      field.classList.add('is-valid');
      field.classList.remove('is-invalid');
    }
  }

  /**
   * Format phone number as user types
   */
  function formatPhoneNumber(field) {
    let value = field.value.replace(/\D/g, '');
    
    // South African phone number formatting
    if (value.startsWith('27')) {
      value = value.replace(/^27(\d{2})(\d{3})(\d{4})$/, '+27 $1 $2 $3');
    } else if (value.startsWith('0')) {
      value = value.replace(/^0(\d{2})(\d{3})(\d{4})$/, '0$1 $2 $3');
    }
    
    field.value = value;
  }

  /**
   * Show field success state
   */
  function showFieldSuccess(field) {
    const feedback = field.parentNode.querySelector('.valid-feedback');
    if (feedback) {
      feedback.style.display = 'block';
    }
    
    // Hide invalid feedback
    const invalidFeedback = field.parentNode.querySelector('.invalid-feedback');
    if (invalidFeedback) {
      invalidFeedback.style.display = 'none';
    }
  }

  /**
   * Show field error state
   */
  function showFieldError(field, message) {
    const feedback = field.parentNode.querySelector('.invalid-feedback');
    if (feedback) {
      feedback.textContent = message;
      feedback.style.display = 'block';
    }
    
    // Hide valid feedback
    const validFeedback = field.parentNode.querySelector('.valid-feedback');
    if (validFeedback) {
      validFeedback.style.display = 'none';
    }
  }

  /**
   * Initialize form progress indicator
   */
  function initFormProgress(form) {
    const progressContainer = form.querySelector('.form-progress');
    if (!progressContainer) {
      // Create progress indicator
      const progress = document.createElement('div');
      progress.className = 'form-progress';
      progress.innerHTML = `
        <div class="progress-bar">
          <div class="progress-fill" style="width: 0%"></div>
        </div>
        <div class="progress-text">
          <span class="completed-fields">0</span> of <span class="total-fields">0</span> fields completed
        </div>
      `;
      form.insertBefore(progress, form.firstChild);
    }
    
    updateFormProgress(form);
  }

  /**
   * Update form progress indicator
   */
  function updateFormProgress(form) {
    const progressContainer = form.querySelector('.form-progress');
    if (!progressContainer) return;

    const requiredFields = form.querySelectorAll('[required]');
    const validFields = form.querySelectorAll('[required].is-valid');
    
    const total = requiredFields.length;
    const completed = validFields.length;
    const percentage = total > 0 ? (completed / total) * 100 : 0;

    const progressFill = progressContainer.querySelector('.progress-fill');
    const completedSpan = progressContainer.querySelector('.completed-fields');
    const totalSpan = progressContainer.querySelector('.total-fields');

    if (progressFill) progressFill.style.width = percentage + '%';
    if (completedSpan) completedSpan.textContent = completed;
    if (totalSpan) totalSpan.textContent = total;

    // Update progress bar color based on completion
    if (progressFill) {
      if (percentage === 100) {
        progressFill.style.backgroundColor = 'var(--color-success)';
      } else if (percentage >= 50) {
        progressFill.style.backgroundColor = 'var(--color-primary-gold)';
      } else {
        progressFill.style.backgroundColor = 'var(--color-gray-light)';
      }
    }
  }

  /**
   * Show validation summary for form
   */
  function showValidationSummary(form) {
    const invalidFields = form.querySelectorAll('.is-invalid');
    if (invalidFields.length === 0) return;

    let summary = `Please correct the following ${invalidFields.length} error${invalidFields.length > 1 ? 's' : ''}:\n\n`;
    
    invalidFields.forEach(function(field, index) {
      const label = field.closest('.form-group')?.querySelector('label')?.textContent || 'Field';
      const feedback = field.parentNode.querySelector('.invalid-feedback')?.textContent || 'Invalid input';
      summary += `${index + 1}. ${label.replace('*', '').trim()}: ${feedback}\n`;
    });

    // Show summary in a more user-friendly way
    showNotification(summary, 'error', 8000);
    
    // Focus on first invalid field
    invalidFields[0].focus();
  }

  /**
   * Enhanced form submission with better UX
   */
  function handleFormSubmission(form) {
    const submitBtn = form.querySelector('button[type="submit"]');
    const originalText = submitBtn?.innerHTML || 'Submit';
    
    // Show loading state
    if (submitBtn) {
      submitBtn.disabled = true;
      submitBtn.innerHTML = `
        <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor" style="margin-right: 8px; animation: spin 1s linear infinite;">
          <path d="M12,4V2A10,10 0 0,0 2,12H4A8,8 0 0,1 12,4Z"/>
        </svg>
        Sending...
      `;
    }
    
    // Mock API call with realistic delay
    setTimeout(function() {
      // Reset button
      if (submitBtn) {
        submitBtn.disabled = false;
        submitBtn.innerHTML = originalText;
      }
      
      // Show success message
      showNotification('Thank you for your message! We will get back to you within 24 hours.', 'success', 5000);
      
      // Reset form with animation
      resetFormWithAnimation(form);
      
    }, 2000);
  }

  /**
   * Reset form with smooth animation
   */
  function resetFormWithAnimation(form) {
    // Fade out form content
    form.style.transition = 'opacity 0.3s ease';
    form.style.opacity = '0.5';
    
    setTimeout(function() {
      // Reset form
      form.reset();
      
      // Clear validation states
      const inputs = form.querySelectorAll('.form-control, .form-check-input');
      inputs.forEach(clearValidationState);
      
      // Update progress
      updateFormProgress(form);
      
      // Fade form back in
      form.style.opacity = '1';
    }, 300);
  }

  /**
   * Show notification with enhanced styling
   */
  function showNotification(message, type = 'info', duration = 5000) {
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    
    let icon = '';
    switch (type) {
      case 'success':
        icon = '<svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z"/></svg>';
        break;
      case 'error':
        icon = '<svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-2h2v2zm0-4h-2V7h2v6z"/></svg>';
        break;
      case 'warning':
        icon = '<svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor"><path d="M1 21h22L12 2 1 21zm12-3h-2v-2h2v2zm0-4h-2v-4h2v4z"/></svg>';
        break;
      default:
        icon = '<svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-6h2v6zm0-8h-2V7h2v2z"/></svg>';
    }
    
    notification.innerHTML = `
      <div class="notification-icon">${icon}</div>
      <div class="notification-content">
        <div class="notification-message">${message}</div>
      </div>
      <button class="notification-close" aria-label="Close notification">
        <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
          <path d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z"/>
        </svg>
      </button>
    `;
    
    notification.style.cssText = `
      position: fixed;
      top: 20px;
      right: 20px;
      max-width: 400px;
      background-color: ${getNotificationColor(type)};
      color: white;
      padding: 16px;
      border-radius: 8px;
      box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
      z-index: 10000;
      display: flex;
      align-items: flex-start;
      gap: 12px;
      transform: translateX(100%);
      transition: transform 0.3s ease;
    `;
    
    // Close button functionality
    const closeBtn = notification.querySelector('.notification-close');
    closeBtn.addEventListener('click', function() {
      hideNotification(notification);
    });
    
    document.body.appendChild(notification);
    
    // Animate in
    setTimeout(() => {
      notification.style.transform = 'translateX(0)';
    }, 100);
    
    // Auto-hide
    setTimeout(() => {
      hideNotification(notification);
    }, duration);
  }

  /**
   * Get notification color based on type
   */
  function getNotificationColor(type) {
    switch (type) {
      case 'success': return 'var(--color-success)';
      case 'error': return 'var(--color-error)';
      case 'warning': return 'var(--color-warning)';
      default: return 'var(--color-gray-dark)';
    }
  }

  /**
   * Hide notification with animation
   */
  function hideNotification(notification) {
    notification.style.transform = 'translateX(100%)';
    setTimeout(() => {
      if (document.body.contains(notification)) {
        document.body.removeChild(notification);
      }
    }, 300);
  }

  /**
   * Utility function to debounce events
   */
  function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
      const later = function() {
        clearTimeout(timeout);
        func(...args);
      };
      clearTimeout(timeout);
      timeout = setTimeout(later, wait);
    };
  }

  /**
   * Initialize local storage for form persistence
   */
  function initFormPersistence() {
    const forms = document.querySelectorAll('form[data-persist="true"], .application-step-form, .contact-form');
    
    forms.forEach(function(form) {
      const formId = form.id || generateFormId(form);
      const storageKey = `lcpff-form-${formId}`;
      
      // Load saved data on page load
      loadFormData(form, storageKey);
      
      // Save data on input changes
      const inputs = form.querySelectorAll('input, textarea, select');
      inputs.forEach(function(input) {
        input.addEventListener('input', debounce(function() {
          saveFormData(form, storageKey);
        }, 500));
        
        input.addEventListener('change', function() {
          saveFormData(form, storageKey);
        });
      });
      
      // Clear data on successful submission
      form.addEventListener('submit', function() {
        setTimeout(function() {
          clearFormData(storageKey);
        }, 3000); // Clear after successful submission
      });
      
      // Add form persistence indicator
      addPersistenceIndicator(form);
    });
    
    // Initialize multi-step form progress tracking
    initMultiStepProgress();
    
    // Initialize data cleanup
    initDataCleanup();
  }

  /**
   * Generate unique form ID
   */
  function generateFormId(form) {
    const formName = form.className.replace(/\s+/g, '-') || 'form';
    const timestamp = Date.now();
    return `${formName}-${timestamp}`;
  }

  /**
   * Save form data to localStorage
   */
  function saveFormData(form, storageKey) {
    try {
      const formData = {};
      const inputs = form.querySelectorAll('input, textarea, select');
      
      inputs.forEach(function(input) {
        const name = input.name || input.id;
        if (!name) return;
        
        if (input.type === 'checkbox') {
          if (!formData[name]) formData[name] = [];
          if (input.checked) {
            formData[name].push(input.value);
          }
        } else if (input.type === 'radio') {
          if (input.checked) {
            formData[name] = input.value;
          }
        } else if (input.type === 'file') {
          // Don't persist file inputs for security reasons
          return;
        } else {
          formData[name] = input.value;
        }
      });
      
      const persistenceData = {
        data: formData,
        timestamp: Date.now(),
        url: window.location.href,
        formId: form.id || 'unknown'
      };
      
      storage.setItem(storageKey, JSON.stringify(persistenceData));
      
      // Show save indicator
      showSaveIndicator(form);
      
    } catch (error) {
      console.warn('Failed to save form data:', error);
    }
  }

  /**
   * Load form data from localStorage
   */
  function loadFormData(form, storageKey) {
    try {
      const savedData = storage.getItem(storageKey);
      if (!savedData) return;
      
      const persistenceData = JSON.parse(savedData);
      const formData = persistenceData.data;
      
      // Check if data is not too old (7 days)
      const maxAge = 7 * 24 * 60 * 60 * 1000; // 7 days in milliseconds
      if (Date.now() - persistenceData.timestamp > maxAge) {
        storage.removeItem(storageKey);
        return;
      }
      
      // Restore form values
      Object.keys(formData).forEach(function(name) {
        const value = formData[name];
        const inputs = form.querySelectorAll(`[name="${name}"], #${name}`);
        
        inputs.forEach(function(input) {
          if (input.type === 'checkbox') {
            input.checked = Array.isArray(value) ? value.includes(input.value) : value === input.value;
          } else if (input.type === 'radio') {
            input.checked = input.value === value;
          } else if (input.tagName === 'SELECT') {
            input.value = value;
          } else if (input.type !== 'file') {
            input.value = value;
          }
          
          // Trigger change event to update any dependent UI
          input.dispatchEvent(new Event('change', { bubbles: true }));
        });
      });
      
      // Show restoration indicator
      showRestorationIndicator(form, persistenceData.timestamp);
      
    } catch (error) {
      console.warn('Failed to load form data:', error);
      storage.removeItem(storageKey);
    }
  }

  /**
   * Clear form data from localStorage
   */
  function clearFormData(storageKey) {
    try {
      storage.removeItem(storageKey);
    } catch (error) {
      console.warn('Failed to clear form data:', error);
    }
  }

  /**
   * Add persistence indicator to form
   */
  function addPersistenceIndicator(form) {
    const indicator = document.createElement('div');
    indicator.className = 'form-persistence-indicator';
    indicator.innerHTML = `
      <div class="persistence-status">
        <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor" class="save-icon">
          <path d="M17 3H5c-1.11 0-2 .9-2 2v14c0 1.1.89 2 2 2h14c1.1 0 2-.9 2-2V7l-4-4zm-5 16c-1.66 0-3-1.34-3-3s1.34-3 3-3 3 1.34 3 3-1.34 3-3 3zm3-10H5V5h10v4z"/>
        </svg>
        <span class="status-text">Auto-saving...</span>
      </div>
    `;
    
    indicator.style.cssText = `
      position: absolute;
      top: -30px;
      right: 0;
      font-size: 12px;
      color: var(--color-gray-medium);
      opacity: 0;
      transition: opacity 0.3s ease;
      display: flex;
      align-items: center;
      gap: 4px;
    `;
    
    // Position form relatively if not already
    if (getComputedStyle(form).position === 'static') {
      form.style.position = 'relative';
    }
    
    form.appendChild(indicator);
  }

  /**
   * Show save indicator
   */
  function showSaveIndicator(form) {
    const indicator = form.querySelector('.form-persistence-indicator');
    if (!indicator) return;
    
    const statusText = indicator.querySelector('.status-text');
    const saveIcon = indicator.querySelector('.save-icon');
    
    statusText.textContent = 'Saved';
    saveIcon.style.color = 'var(--color-success)';
    indicator.style.opacity = '1';
    
    setTimeout(function() {
      indicator.style.opacity = '0';
    }, 2000);
  }

  /**
   * Show restoration indicator
   */
  function showRestorationIndicator(form, timestamp) {
    const timeAgo = getTimeAgo(timestamp);
    
    showNotification(
      `Form data restored from ${timeAgo}. Continue where you left off!`,
      'info',
      5000
    );
  }

  /**
   * Get human-readable time ago
   */
  function getTimeAgo(timestamp) {
    const now = Date.now();
    const diff = now - timestamp;
    
    const minutes = Math.floor(diff / (1000 * 60));
    const hours = Math.floor(diff / (1000 * 60 * 60));
    const days = Math.floor(diff / (1000 * 60 * 60 * 24));
    
    if (days > 0) {
      return `${days} day${days > 1 ? 's' : ''} ago`;
    } else if (hours > 0) {
      return `${hours} hour${hours > 1 ? 's' : ''} ago`;
    } else if (minutes > 0) {
      return `${minutes} minute${minutes > 1 ? 's' : ''} ago`;
    } else {
      return 'just now';
    }
  }

  /**
   * Initialize multi-step form progress tracking
   */
  function initMultiStepProgress() {
    const multiStepForms = document.querySelectorAll('.multi-step-form, .application-form-container');
    
    multiStepForms.forEach(function(form) {
      const progressKey = `lcpff-progress-${form.id || 'multi-step'}`;
      
      // Load saved progress
      loadMultiStepProgress(form, progressKey);
      
      // Save progress on step changes
      const stepButtons = form.querySelectorAll('.next-step, .prev-step, .step-indicator');
      stepButtons.forEach(function(button) {
        button.addEventListener('click', function() {
          setTimeout(function() {
            saveMultiStepProgress(form, progressKey);
          }, 100);
        });
      });
    });
  }

  /**
   * Save multi-step form progress
   */
  function saveMultiStepProgress(form, progressKey) {
    try {
      const activeStep = form.querySelector('.form-step.active');
      const currentStepNumber = activeStep ? activeStep.dataset.step || 1 : 1;
      
      const progressData = {
        currentStep: parseInt(currentStepNumber),
        timestamp: Date.now(),
        url: window.location.href,
        completedSteps: getCompletedSteps(form)
      };
      
      storage.setItem(progressKey, JSON.stringify(progressData));
      
    } catch (error) {
      console.warn('Failed to save multi-step progress:', error);
    }
  }

  /**
   * Load multi-step form progress
   */
  function loadMultiStepProgress(form, progressKey) {
    try {
      const savedProgress = storage.getItem(progressKey);
      if (!savedProgress) return;
      
      const progressData = JSON.parse(savedProgress);
      
      // Check if progress is not too old (24 hours)
      const maxAge = 24 * 60 * 60 * 1000; // 24 hours
      if (Date.now() - progressData.timestamp > maxAge) {
        storage.removeItem(progressKey);
        return;
      }
      
      // Show option to continue from saved step
      showProgressRestoreOption(form, progressData);
      
    } catch (error) {
      console.warn('Failed to load multi-step progress:', error);
      storage.removeItem(progressKey);
    }
  }

  /**
   * Get completed steps from form
   */
  function getCompletedSteps(form) {
    const completedSteps = [];
    const steps = form.querySelectorAll('.form-step');
    
    steps.forEach(function(step, index) {
      const stepNumber = parseInt(step.dataset.step || index + 1);
      const requiredFields = step.querySelectorAll('[required]');
      const validFields = step.querySelectorAll('[required].is-valid');
      
      if (requiredFields.length === 0 || requiredFields.length === validFields.length) {
        completedSteps.push(stepNumber);
      }
    });
    
    return completedSteps;
  }

  /**
   * Show progress restore option
   */
  function showProgressRestoreOption(form, progressData) {
    const modal = document.createElement('div');
    modal.className = 'progress-restore-modal';
    modal.innerHTML = `
      <div class="modal-overlay"></div>
      <div class="modal-content">
        <h3>Continue Previous Application?</h3>
        <p>We found a saved application from ${getTimeAgo(progressData.timestamp)}. Would you like to continue where you left off?</p>
        <div class="modal-actions">
          <button class="btn btn-primary" id="continue-progress">Continue</button>
          <button class="btn btn-secondary" id="start-fresh">Start Fresh</button>
        </div>
      </div>
    `;
    
    modal.style.cssText = `
      position: fixed;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      z-index: 10000;
      display: flex;
      align-items: center;
      justify-content: center;
    `;
    
    document.body.appendChild(modal);
    
    // Handle continue button
    modal.querySelector('#continue-progress').addEventListener('click', function() {
      if (window.LCPFF && window.LCPFF.application && window.LCPFF.application.navigateToStep) {
        window.LCPFF.application.navigateToStep(progressData.currentStep);
      }
      document.body.removeChild(modal);
    });
    
    // Handle start fresh button
    modal.querySelector('#start-fresh').addEventListener('click', function() {
      const progressKey = `lcpff-progress-${form.id || 'multi-step'}`;
      storage.removeItem(progressKey);
      document.body.removeChild(modal);
    });
    
    // Close on overlay click
    modal.querySelector('.modal-overlay').addEventListener('click', function() {
      document.body.removeChild(modal);
    });
  }

  /**
   * Initialize data cleanup
   */
  function initDataCleanup() {
    // Clean up old form data on page load
    cleanupOldFormData();
    
    // Set up periodic cleanup
    setInterval(cleanupOldFormData, 60 * 60 * 1000); // Every hour
  }

  /**
   * Clean up old form data from localStorage
   */
  function cleanupOldFormData() {
    try {
      const maxAge = 7 * 24 * 60 * 60 * 1000; // 7 days
      const keysToRemove = [];
      
      for (let i = 0; i < storage.length; i++) {
        const key = storage.key(i);
        if (key && key.startsWith('lcpff-')) {
          try {
            const data = JSON.parse(storage.getItem(key));
            if (data.timestamp && Date.now() - data.timestamp > maxAge) {
              keysToRemove.push(key);
            }
          } catch (error) {
            // Invalid data, mark for removal
            keysToRemove.push(key);
          }
        }
      }
      
      keysToRemove.forEach(function(key) {
        storage.removeItem(key);
      });
      
      if (keysToRemove.length > 0) {
        console.log(`Cleaned up ${keysToRemove.length} old form data entries`);
      }
      
    } catch (error) {
      console.warn('Failed to cleanup old form data:', error);
    }
  }

  /**
   * Get storage usage information
   */
  function getStorageInfo() {
    try {
      let totalSize = 0;
      let lcpffSize = 0;
      
      for (let key in localStorage) {
        if (storage.hasOwnProperty(key)) {
          const size = storage[key].length;
          totalSize += size;
          
          if (key.startsWith('lcpff-')) {
            lcpffSize += size;
          }
        }
      }
      
      return {
        total: totalSize,
        lcpff: lcpffSize,
        available: 5 * 1024 * 1024 - totalSize // Approximate 5MB limit
      };
    } catch (error) {
      return { total: 0, lcpff: 0, available: 0 };
    }
  }

  /**
   * Handle storage quota exceeded
   */
  function handleStorageQuotaExceeded() {
    showNotification(
      'Storage space is full. Some form data may not be saved. Please clear your browser data or complete your application.',
      'warning',
      10000
    );
    
    // Try to free up space by removing oldest entries
    cleanupOldFormData();
  }

  // Export form persistence functions
  window.LCPFF = window.LCPFF || {};
  window.LCPFF.formPersistence = {
    saveFormData: saveFormData,
    loadFormData: loadFormData,
    clearFormData: clearFormData,
    getStorageInfo: getStorageInfo
  };

})();
  /**
   * Application Portal Functionality
   */
  function initApplicationPortal() {
    const startButton = document.querySelector('#start-application');
    const applicationForm = document.querySelector('#application-form');
    const formSteps = document.querySelectorAll('.form-step');
    const progressFill = document.querySelector('.progress-fill');
    const currentStepSpan = document.querySelector('.current-step');
    
    if (!startButton || !applicationForm) return;

    let currentStep = 1;
    const totalSteps = 6;

    // Start application
    startButton.addEventListener('click', function(event) {
      event.preventDefault();
      showApplicationForm();
      updateProgress();
    });

    // Show application form
    function showApplicationForm() {
      applicationForm.style.display = 'block';
      applicationForm.scrollIntoView({ behavior: 'smooth' });
    }

    // Update progress bar
    function updateProgress() {
      const progressPercentage = (currentStep / totalSteps) * 100;
      progressFill.style.width = progressPercentage + '%';
      currentStepSpan.textContent = 'Step ' + currentStep;
    }

    // Handle form submissions for each step
    const stepForms = document.querySelectorAll('.application-step-form');
    stepForms.forEach(function(form, index) {
      form.addEventListener('submit', function(event) {
        event.preventDefault();
        
        if (validateForm(form)) {
          saveFormData(form);
          nextStep();
        }
      });
    });

    // Next step function
    function nextStep() {
      if (currentStep < totalSteps) {
        // Hide current step
        const currentStepElement = document.getElementById('step-' + currentStep);
        if (currentStepElement) {
          currentStepElement.style.display = 'none';
          currentStepElement.classList.remove('active');
        }

        // Show next step
        currentStep++;
        const nextStepElement = document.getElementById('step-' + currentStep);
        if (nextStepElement) {
          nextStepElement.style.display = 'block';
          nextStepElement.classList.add('active');
          nextStepElement.scrollIntoView({ behavior: 'smooth' });
        }

        updateProgress();
      }
    }

    // Previous step function
    function previousStep() {
      if (currentStep > 1) {
        // Hide current step
        const currentStepElement = document.getElementById('step-' + currentStep);
        if (currentStepElement) {
          currentStepElement.style.display = 'none';
          currentStepElement.classList.remove('active');
        }

        // Show previous step
        currentStep--;
        const prevStepElement = document.getElementById('step-' + currentStep);
        if (prevStepElement) {
          prevStepElement.style.display = 'block';
          prevStepElement.classList.add('active');
          prevStepElement.scrollIntoView({ behavior: 'smooth' });
        }

        updateProgress();
      }
    }

    // Save form data to localStorage
    function saveFormData(form) {
      const formData = new FormData(form);
      const data = {};
      
      for (let [key, value] of formData.entries()) {
        data[key] = value;
      }
      
      const existingData = JSON.parse(storage.getItem('lcpff-application') || '{}');
      const updatedData = { ...existingData, ...data };
      
      storage.setItem('lcpff-application', JSON.stringify(updatedData));
      
      // Show save confirmation
      showSaveConfirmation();
    }

    // Show save confirmation
    function showSaveConfirmation() {
      // Create temporary notification
      const notification = document.createElement('div');
      notification.className = 'save-notification';
      notification.textContent = 'Progress saved successfully!';
      notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background-color: var(--color-success);
        color: white;
        padding: 12px 20px;
        border-radius: 4px;
        z-index: 1000;
        font-size: 14px;
        box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        transition: all 0.3s ease;
      `;
      
      document.body.appendChild(notification);
      
      setTimeout(function() {
        notification.style.opacity = '0';
        setTimeout(function() {
          document.body.removeChild(notification);
        }, 300);
      }, 3000);
    }

    // Load saved form data
    function loadSavedData() {
      const savedData = JSON.parse(storage.getItem('lcpff-application') || '{}');
      
      Object.keys(savedData).forEach(function(key) {
        const field = document.querySelector('[name="' + key + '"]');
        if (field) {
          field.value = savedData[key];
        }
      });
    }

    // Save draft functionality
    const saveDraftButtons = document.querySelectorAll('#save-draft');
    saveDraftButtons.forEach(function(button) {
      button.addEventListener('click', function() {
        const form = button.closest('form');
        if (form) {
          saveFormData(form);
        }
      });
    });

    // Load saved data on page load
    loadSavedData();

    // Add navigation functionality to stepper
    const stepItems = document.querySelectorAll('.step-item');
    stepItems.forEach(function(item, index) {
      item.addEventListener('click', function() {
        const targetStep = index + 1;
        if (targetStep <= currentStep) {
          navigateToStep(targetStep);
        }
      });
    });

    // Navigate to specific step
    function navigateToStep(stepNumber) {
      if (stepNumber >= 1 && stepNumber <= totalSteps) {
        // Hide current step
        const currentStepElement = document.getElementById('step-' + currentStep);
        if (currentStepElement) {
          currentStepElement.style.display = 'none';
          currentStepElement.classList.remove('active');
        }

        // Show target step
        currentStep = stepNumber;
        const targetStepElement = document.getElementById('step-' + currentStep);
        if (targetStepElement) {
          targetStepElement.style.display = 'block';
          targetStepElement.classList.add('active');
          targetStepElement.scrollIntoView({ behavior: 'smooth' });
        }

        updateProgress();
      }
    }

    // Export functions for potential use
    window.LCPFF = window.LCPFF || {};
    window.LCPFF.application = {
      nextStep: nextStep,
      previousStep: previousStep,
      navigateToStep: navigateToStep,
      saveFormData: saveFormData,
      loadSavedData: loadSavedData
    };
  }

  // Initialize application portal when DOM is loaded
  document.addEventListener('DOMContentLoaded', function() {
    initApplicationPortal();
  });
  /**
   * Initialize character counters
   */
  function initCharacterCounters() {
    const textareas = document.querySelectorAll('textarea[id="project-logline"], textarea[id="project-synopsis"]');
    
    textareas.forEach(function(textarea) {
      const counter = textarea.parentNode.querySelector('.character-counter');
      if (!counter) return;

      const currentCount = counter.querySelector('.current-count');
      const maxCount = counter.querySelector('.max-count');
      
      function updateCounter() {
        let count;
        if (textarea.id === 'project-logline') {
          count = textarea.value.length;
          const maxLength = 150;
          currentCount.textContent = count;
          
          if (count > maxLength) {
            counter.classList.add('over-limit');
          } else {
            counter.classList.remove('over-limit');
          }
        } else if (textarea.id === 'project-synopsis') {
          // Word count for synopsis
          count = textarea.value.trim().split(/\s+/).filter(word => word.length > 0).length;
          currentCount.textContent = count;
        }
      }

      textarea.addEventListener('input', updateCounter);
      updateCounter(); // Initial count
    });
  }

  /**
   * Initialize checkbox group validation
   */
  function initCheckboxValidation() {
    const checkboxGroups = document.querySelectorAll('.checkbox-group');
    
    checkboxGroups.forEach(function(group) {
      const checkboxes = group.querySelectorAll('input[type="checkbox"]');
      const groupName = checkboxes[0] ? checkboxes[0].name : '';
      
      checkboxes.forEach(function(checkbox) {
        checkbox.addEventListener('change', function() {
          validateCheckboxGroup(groupName);
        });
      });
    });
  }

  /**
   * Validate checkbox group
   */
  function validateCheckboxGroup(groupName) {
    const checkboxes = document.querySelectorAll('input[name="' + groupName + '"]');
    const isChecked = Array.from(checkboxes).some(cb => cb.checked);
    
    checkboxes.forEach(function(checkbox) {
      if (isChecked) {
        checkbox.classList.remove('is-invalid');
        checkbox.classList.add('is-valid');
      } else {
        checkbox.classList.add('is-invalid');
        checkbox.classList.remove('is-valid');
      }
    });
    
    return isChecked;
  }

  /**
   * Enhanced form validation for Step 2
   */
  function validateProjectForm(form) {
    let isValid = true;
    
    // Standard field validation
    const requiredFields = form.querySelectorAll('[required]');
    requiredFields.forEach(function(field) {
      if (!validateField(field)) {
        isValid = false;
      }
    });
    
    // Checkbox group validation
    const targetAudienceValid = validateCheckboxGroup('targetAudience');
    if (!targetAudienceValid) {
      isValid = false;
    }
    
    // Logline character limit validation
    const logline = form.querySelector('#project-logline');
    if (logline && logline.value.length > 150) {
      logline.classList.add('is-invalid');
      isValid = false;
    }
    
    return isValid;
  }

  // Initialize new functionality when DOM is loaded
  document.addEventListener('DOMContentLoaded', function() {
    initCharacterCounters();
    initCheckboxValidation();
    
    // Override form validation for project form
    const projectForm = document.getElementById('project-form');
    if (projectForm) {
      projectForm.addEventListener('submit', function(event) {
        event.preventDefault();
        
        if (validateProjectForm(projectForm)) {
          // Save form data and proceed to next step
          const formData = new FormData(projectForm);
          const data = {};
          
          // Handle checkbox groups
          const checkboxes = projectForm.querySelectorAll('input[type="checkbox"]:checked');
          const targetAudience = [];
          checkboxes.forEach(function(checkbox) {
            if (checkbox.name === 'targetAudience') {
              targetAudience.push(checkbox.value);
            }
          });
          
          // Add checkbox data
          if (targetAudience.length > 0) {
            data.targetAudience = targetAudience;
          }
          
          // Add other form data
          for (let [key, value] of formData.entries()) {
            if (key !== 'targetAudience') {
              data[key] = value;
            }
          }
          
          // Save to localStorage
          const existingData = JSON.parse(storage.getItem('lcpff-application') || '{}');
          const updatedData = { ...existingData, ...data };
          storage.setItem('lcpff-application', JSON.stringify(updatedData));
          
          // Show save confirmation and proceed
          showSaveConfirmation();
          if (window.LCPFF && window.LCPFF.application) {
            window.LCPFF.application.nextStep();
          }
        }
      });
    }
  });
  /**
   * Initialize file upload functionality
   */
  function initFileUploads() {
    const uploadAreas = document.querySelectorAll('.file-upload-area');
    
    uploadAreas.forEach(function(area) {
      const uploadZone = area.querySelector('.upload-zone');
      const fileInput = area.querySelector('.file-input');
      const browseButton = area.querySelector('.upload-browse');
      const preview = area.querySelector('.upload-preview');
      const uploadType = area.getAttribute('data-upload');
      
      if (!uploadZone || !fileInput) return;

      // Browse button click
      if (browseButton) {
        browseButton.addEventListener('click', function(event) {
          event.preventDefault();
          fileInput.click();
        });
      }

      // Upload zone click
      uploadZone.addEventListener('click', function() {
        fileInput.click();
      });

      // File input change
      fileInput.addEventListener('change', function() {
        handleFileSelection(fileInput, area, preview, uploadType);
      });

      // Drag and drop events
      uploadZone.addEventListener('dragover', function(event) {
        event.preventDefault();
        area.classList.add('dragover');
      });

      uploadZone.addEventListener('dragleave', function(event) {
        event.preventDefault();
        area.classList.remove('dragover');
      });

      uploadZone.addEventListener('drop', function(event) {
        event.preventDefault();
        area.classList.remove('dragover');
        
        const files = event.dataTransfer.files;
        if (files.length > 0) {
          fileInput.files = files;
          handleFileSelection(fileInput, area, preview, uploadType);
        }
      });
    });
  }

  /**
   * Handle file selection
   */
  function handleFileSelection(fileInput, area, preview, uploadType) {
    const files = Array.from(fileInput.files);
    
    if (files.length === 0) return;

    // Validate files
    const validFiles = validateFiles(files, uploadType);
    
    if (validFiles.length === 0) {
      area.classList.add('error');
      setTimeout(function() {
        area.classList.remove('error');
      }, 3000);
      return;
    }

    // Show preview
    showFilePreview(validFiles, preview, uploadType);
    area.classList.add('success');
    
    // Simulate upload progress
    simulateUpload(area, preview);
  }

  /**
   * Validate uploaded files
   */
  function validateFiles(files, uploadType) {
    const validFiles = [];
    
    files.forEach(function(file) {
      let isValid = true;
      let maxSize = 10 * 1024 * 1024; // 10MB default
      let allowedTypes = [];

      // Set validation rules based on upload type
      switch (uploadType) {
        case 'script':
        case 'statement':
          allowedTypes = ['application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'];
          maxSize = uploadType === 'script' ? 10 * 1024 * 1024 : 5 * 1024 * 1024;
          break;
        case 'visual':
        case 'additional':
          allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'application/pdf'];
          maxSize = 5 * 1024 * 1024;
          break;
        case 'portfolio':
          allowedTypes = ['video/mp4', 'video/quicktime', 'application/pdf'];
          maxSize = file.type.startsWith('video/') ? 100 * 1024 * 1024 : 10 * 1024 * 1024;
          break;
      }

      // Check file type
      if (!allowedTypes.includes(file.type)) {
        isValid = false;
        showFileError('Invalid file type: ' + file.name);
      }

      // Check file size
      if (file.size > maxSize) {
        isValid = false;
        showFileError('File too large: ' + file.name + ' (Max: ' + formatFileSize(maxSize) + ')');
      }

      if (isValid) {
        validFiles.push(file);
      }
    });

    return validFiles;
  }

  /**
   * Show file preview
   */
  function showFilePreview(files, preview, uploadType) {
    if (!preview) return;

    preview.style.display = 'block';
    
    if (files.length === 1) {
      // Single file preview
      const filePreview = preview.querySelector('.file-preview');
      if (filePreview) {
        updateFilePreview(filePreview, files[0]);
      }
    } else {
      // Multiple files preview
      const fileList = preview.querySelector('.file-list');
      if (fileList) {
        fileList.innerHTML = '';
        files.forEach(function(file) {
          const filePreview = createFilePreview(file);
          fileList.appendChild(filePreview);
        });
      }
    }
  }

  /**
   * Update file preview
   */
  function updateFilePreview(previewElement, file) {
    const fileName = previewElement.querySelector('.file-name');
    const fileSize = previewElement.querySelector('.file-size');
    const removeButton = previewElement.querySelector('.file-remove');

    if (fileName) fileName.textContent = file.name;
    if (fileSize) fileSize.textContent = formatFileSize(file.size);

    if (removeButton) {
      removeButton.addEventListener('click', function() {
        previewElement.closest('.upload-preview').style.display = 'none';
        const area = previewElement.closest('.file-upload-area');
        const fileInput = area.querySelector('.file-input');
        if (fileInput) fileInput.value = '';
        area.classList.remove('success');
      });
    }
  }

  /**
   * Create file preview element
   */
  function createFilePreview(file) {
    const preview = document.createElement('div');
    preview.className = 'file-preview';
    
    preview.innerHTML = `
      <div class="file-info">
        <div class="file-icon">
          <svg width="24" height="24" viewBox="0 0 24 24" fill="currentColor">
            <path d="M14,2H6A2,2 0 0,0 4,4V20A2,2 0 0,0 6,22H18A2,2 0 0,0 20,20V8L14,2M18,20H6V4H13V9H18V20Z"/>
          </svg>
        </div>
        <div class="file-details">
          <div class="file-name">${file.name}</div>
          <div class="file-size">${formatFileSize(file.size)}</div>
        </div>
      </div>
      <button type="button" class="file-remove" aria-label="Remove file">
        <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">
          <path d="M19,6.41L17.59,5L12,10.59L6.41,5L5,6.41L10.59,12L5,17.59L6.41,19L12,13.41L17.59,19L19,17.59L13.41,12L19,6.41Z"/>
        </svg>
      </button>
    `;

    // Add remove functionality
    const removeButton = preview.querySelector('.file-remove');
    removeButton.addEventListener('click', function() {
      preview.remove();
      
      // Check if file list is empty
      const fileList = preview.closest('.file-list');
      if (fileList && fileList.children.length === 0) {
        const uploadPreview = fileList.closest('.upload-preview');
        if (uploadPreview) uploadPreview.style.display = 'none';
        
        const area = uploadPreview.closest('.file-upload-area');
        const fileInput = area.querySelector('.file-input');
        if (fileInput) fileInput.value = '';
        area.classList.remove('success');
      }
    });

    return preview;
  }

  /**
   * Simulate file upload progress
   */
  function simulateUpload(area, preview) {
    const progressElement = preview.querySelector('.upload-progress');
    if (!progressElement) return;

    progressElement.style.display = 'block';
    const progressFill = progressElement.querySelector('.progress-fill');
    const progressPercent = progressElement.querySelector('.progress-percent');

    let progress = 0;
    const interval = setInterval(function() {
      progress += Math.random() * 15;
      if (progress >= 100) {
        progress = 100;
        clearInterval(interval);
        
        setTimeout(function() {
          progressElement.style.display = 'none';
        }, 1000);
      }

      if (progressFill) progressFill.style.width = progress + '%';
      if (progressPercent) progressPercent.textContent = Math.round(progress) + '%';
    }, 200);
  }

  /**
   * Format file size
   */
  function formatFileSize(bytes) {
    if (bytes === 0) return '0 Bytes';
    
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  }

  /**
   * Show file error
   */
  function showFileError(message) {
    // Create temporary error notification
    const notification = document.createElement('div');
    notification.className = 'file-error-notification';
    notification.textContent = message;
    notification.style.cssText = `
      position: fixed;
      top: 20px;
      right: 20px;
      background-color: var(--color-error);
      color: white;
      padding: 12px 20px;
      border-radius: 4px;
      z-index: 1000;
      font-size: 14px;
      box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
      max-width: 300px;
      word-wrap: break-word;
    `;
    
    document.body.appendChild(notification);
    
    setTimeout(function() {
      notification.style.opacity = '0';
      setTimeout(function() {
        if (document.body.contains(notification)) {
          document.body.removeChild(notification);
        }
      }, 300);
    }, 5000);
  }

  // Initialize file uploads when DOM is loaded
  document.addEventListener('DOMContentLoaded', function() {
    initFileUploads();
  });
  /**
   * Initialize financial form functionality
   */
  function initFinancialForm() {
    const totalBudgetInput = document.getElementById('total-budget');
    const fundingRequestedInput = document.getElementById('funding-requested');
    const fundingPercentageInput = document.getElementById('funding-percentage');
    const coFinancingInput = document.getElementById('co-financing');
    const addBudgetItemBtn = document.getElementById('add-budget-item');
    const cashFlowSelect = document.getElementById('cash-flow-needs');
    const customScheduleDiv = document.getElementById('custom-schedule');

    if (!totalBudgetInput || !fundingRequestedInput) return;

    // Calculate funding percentage and co-financing
    function calculateFinancing() {
      const totalBudget = parseFloat(totalBudgetInput.value) || 0;
      const fundingRequested = parseFloat(fundingRequestedInput.value) || 0;

      if (totalBudget > 0) {
        const percentage = (fundingRequested / totalBudget) * 100;
        const coFinancing = totalBudget - fundingRequested;

        if (fundingPercentageInput) {
          fundingPercentageInput.value = percentage.toFixed(1);
        }
        if (coFinancingInput) {
          coFinancingInput.value = coFinancing.toFixed(0);
        }

        // Validate funding percentage (max 75%)
        if (percentage > 75) {
          fundingRequestedInput.classList.add('is-invalid');
          showValidationMessage('Funding cannot exceed 75% of total budget', 'error');
        } else {
          fundingRequestedInput.classList.remove('is-invalid');
        }

        // Validate minimum co-financing (25%)
        const coFinancingPercentage = (coFinancing / totalBudget) * 100;
        if (coFinancingPercentage < 25) {
          showValidationMessage('Co-financing must be at least 25% of total budget', 'warning');
        }
      }
    }

    // Add event listeners for budget calculations
    totalBudgetInput.addEventListener('input', calculateFinancing);
    fundingRequestedInput.addEventListener('input', calculateFinancing);

    // Budget table functionality
    if (addBudgetItemBtn) {
      addBudgetItemBtn.addEventListener('click', addBudgetItem);
    }

    // Initialize budget calculations
    initBudgetCalculations();

    // Cash flow schedule toggle
    if (cashFlowSelect && customScheduleDiv) {
      cashFlowSelect.addEventListener('change', function() {
        if (this.value === 'custom') {
          customScheduleDiv.style.display = 'block';
          const textarea = customScheduleDiv.querySelector('textarea');
          if (textarea) textarea.required = true;
        } else {
          customScheduleDiv.style.display = 'none';
          const textarea = customScheduleDiv.querySelector('textarea');
          if (textarea) textarea.required = false;
        }
      });
    }
  }

  /**
   * Add budget item to table
   */
  function addBudgetItem() {
    const budgetTableBody = document.getElementById('budget-items');
    if (!budgetTableBody) return;

    const newRow = document.createElement('tr');
    newRow.className = 'budget-item';
    
    newRow.innerHTML = `
      <td>
        <select class="form-control budget-category" name="budgetCategory[]" required>
          <option value="">Select category...</option>
          <option value="development">Development</option>
          <option value="pre-production">Pre-Production</option>
          <option value="production">Production</option>
          <option value="post-production">Post-Production</option>
          <option value="marketing">Marketing & Distribution</option>
          <option value="contingency">Contingency</option>
          <option value="other">Other</option>
        </select>
      </td>
      <td>
        <input type="number" class="form-control budget-amount" name="budgetAmount[]" placeholder="0" min="0" required>
      </td>
      <td>
        <span class="budget-percentage">0%</span>
      </td>
      <td>
        <button type="button" class="btn btn-sm btn-tertiary remove-budget-item">Remove</button>
      </td>
    `;

    budgetTableBody.appendChild(newRow);

    // Add event listeners to new row
    const amountInput = newRow.querySelector('.budget-amount');
    const removeBtn = newRow.querySelector('.remove-budget-item');

    if (amountInput) {
      amountInput.addEventListener('input', calculateBudgetTotals);
    }

    if (removeBtn) {
      removeBtn.addEventListener('click', function() {
        newRow.remove();
        calculateBudgetTotals();
        updateRemoveButtons();
      });
    }

    updateRemoveButtons();
  }

  /**
   * Initialize budget calculations
   */
  function initBudgetCalculations() {
    const budgetAmounts = document.querySelectorAll('.budget-amount');
    const removeButtons = document.querySelectorAll('.remove-budget-item');

    budgetAmounts.forEach(function(input) {
      input.addEventListener('input', calculateBudgetTotals);
    });

    removeButtons.forEach(function(button) {
      button.addEventListener('click', function() {
        const row = button.closest('.budget-item');
        if (row) {
          row.remove();
          calculateBudgetTotals();
          updateRemoveButtons();
        }
      });
    });

    updateRemoveButtons();
    calculateBudgetTotals();
  }

  /**
   * Calculate budget totals and percentages
   */
  function calculateBudgetTotals() {
    const budgetAmounts = document.querySelectorAll('.budget-amount');
    const budgetPercentages = document.querySelectorAll('.budget-percentage');
    const totalAmountSpan = document.getElementById('budget-total-amount');
    const totalPercentageSpan = document.getElementById('budget-total-percentage');
    const totalBudgetInput = document.getElementById('total-budget');
    const budgetValidation = document.getElementById('budget-validation');

    let total = 0;
    const amounts = [];

    // Calculate total
    budgetAmounts.forEach(function(input, index) {
      const amount = parseFloat(input.value) || 0;
      amounts.push(amount);
      total += amount;
    });

    // Update total display
    if (totalAmountSpan) {
      totalAmountSpan.textContent = formatCurrency(total);
    }

    // Calculate and update percentages
    budgetPercentages.forEach(function(span, index) {
      if (amounts[index] && total > 0) {
        const percentage = (amounts[index] / total) * 100;
        span.textContent = percentage.toFixed(1) + '%';
      } else {
        span.textContent = '0%';
      }
    });

    // Validate against total project budget
    const totalProjectBudget = parseFloat(totalBudgetInput?.value) || 0;
    
    if (totalProjectBudget > 0 && budgetValidation) {
      const difference = Math.abs(total - totalProjectBudget);
      const percentageDiff = (difference / totalProjectBudget) * 100;

      if (percentageDiff < 1) { // Within 1%
        showBudgetValidation('Budget breakdown matches total budget', 'success');
      } else if (total > totalProjectBudget) {
        showBudgetValidation('Budget breakdown exceeds total budget by R' + formatCurrency(difference), 'error');
      } else {
        showBudgetValidation('Budget breakdown is R' + formatCurrency(difference) + ' less than total budget', 'warning');
      }
    }

    if (totalPercentageSpan) {
      totalPercentageSpan.textContent = '100%';
    }
  }

  /**
   * Update remove button states
   */
  function updateRemoveButtons() {
    const removeButtons = document.querySelectorAll('.remove-budget-item');
    
    // Disable remove button if only one item remains
    if (removeButtons.length <= 1) {
      removeButtons.forEach(function(button) {
        button.disabled = true;
      });
    } else {
      removeButtons.forEach(function(button) {
        button.disabled = false;
      });
    }
  }

  /**
   * Show budget validation message
   */
  function showBudgetValidation(message, type) {
    const budgetValidation = document.getElementById('budget-validation');
    if (!budgetValidation) return;

    budgetValidation.textContent = message;
    budgetValidation.className = 'validation-message ' + type;
  }

  /**
   * Show validation message
   */
  function showValidationMessage(message, type) {
    // Create temporary validation notification
    const notification = document.createElement('div');
    notification.className = 'validation-notification';
    notification.textContent = message;
    
    let backgroundColor;
    switch (type) {
      case 'error':
        backgroundColor = 'var(--color-error)';
        break;
      case 'warning':
        backgroundColor = 'var(--color-warning)';
        break;
      case 'success':
        backgroundColor = 'var(--color-success)';
        break;
      default:
        backgroundColor = 'var(--color-gray-dark)';
    }
    
    notification.style.cssText = `
      position: fixed;
      top: 20px;
      right: 20px;
      background-color: ${backgroundColor};
      color: ${type === 'warning' ? 'var(--color-primary-black)' : 'white'};
      padding: 12px 20px;
      border-radius: 4px;
      z-index: 1000;
      font-size: 14px;
      box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
      max-width: 300px;
      word-wrap: break-word;
    `;
    
    document.body.appendChild(notification);
    
    setTimeout(function() {
      notification.style.opacity = '0';
      setTimeout(function() {
        if (document.body.contains(notification)) {
          document.body.removeChild(notification);
        }
      }, 300);
    }, 4000);
  }

  /**
   * Format currency
   */
  function formatCurrency(amount) {
    return new Intl.NumberFormat('en-ZA', {
      minimumFractionDigits: 0,
      maximumFractionDigits: 0
    }).format(amount);
  }

  /**
   * Validate financial form
   */
  function validateFinancialForm(form) {
    let isValid = true;
    
    // Standard field validation
    const requiredFields = form.querySelectorAll('[required]');
    requiredFields.forEach(function(field) {
      if (!validateField(field)) {
        isValid = false;
      }
    });
    
    // Budget validation
    const totalBudget = parseFloat(form.querySelector('#total-budget')?.value) || 0;
    const fundingRequested = parseFloat(form.querySelector('#funding-requested')?.value) || 0;
    
    if (totalBudget < 2000000) {
      form.querySelector('#total-budget').classList.add('is-invalid');
      isValid = false;
    }
    
    if (fundingRequested > 50000000) {
      form.querySelector('#funding-requested').classList.add('is-invalid');
      isValid = false;
    }
    
    if (fundingRequested > totalBudget * 0.75) {
      form.querySelector('#funding-requested').classList.add('is-invalid');
      isValid = false;
    }
    
    // Financing sources validation
    const financingSourcesValid = validateCheckboxGroup('financingSources');
    if (!financingSourcesValid) {
      isValid = false;
    }
    
    // Budget items validation
    const budgetAmounts = form.querySelectorAll('.budget-amount');
    let hasBudgetItems = false;
    budgetAmounts.forEach(function(input) {
      if (parseFloat(input.value) > 0) {
        hasBudgetItems = true;
      }
    });
    
    if (!hasBudgetItems) {
      showValidationMessage('Please add at least one budget item with an amount', 'error');
      isValid = false;
    }
    
    return isValid;
  }

  // Initialize financial form when DOM is loaded
  document.addEventListener('DOMContentLoaded', function() {
    initFinancialForm();
    
    // Override form validation for financial form
    const financialForm = document.getElementById('financial-form');
    if (financialForm) {
      financialForm.addEventListener('submit', function(event) {
        event.preventDefault();
        
        if (validateFinancialForm(financialForm)) {
          // Save form data and proceed to next step
          const formData = new FormData(financialForm);
          const data = {};
          
          // Handle checkbox groups
          const checkboxes = financialForm.querySelectorAll('input[type="checkbox"]:checked');
          const financingSources = [];
          checkboxes.forEach(function(checkbox) {
            if (checkbox.name === 'financingSources') {
              financingSources.push(checkbox.value);
            }
          });
          
          // Add checkbox data
          if (financingSources.length > 0) {
            data.financingSources = financingSources;
          }
          
          // Handle budget items
          const budgetCategories = financialForm.querySelectorAll('select[name="budgetCategory[]"]');
          const budgetAmounts = financialForm.querySelectorAll('input[name="budgetAmount[]"]');
          const budgetItems = [];
          
          budgetCategories.forEach(function(select, index) {
            const amount = budgetAmounts[index];
            if (select.value && amount.value) {
              budgetItems.push({
                category: select.value,
                amount: parseFloat(amount.value)
              });
            }
          });
          
          if (budgetItems.length > 0) {
            data.budgetItems = budgetItems;
          }
          
          // Add other form data
          for (let [key, value] of formData.entries()) {
            if (key !== 'financingSources' && !key.includes('budget')) {
              data[key] = value;
            }
          }
          
          // Save to localStorage
          const existingData = JSON.parse(storage.getItem('lcpff-application') || '{}');
          const updatedData = { ...existingData, ...data };
          storage.setItem('lcpff-application', JSON.stringify(updatedData));
          
          // Show save confirmation and proceed
          showSaveConfirmation();
          if (window.LCPFF && window.LCPFF.application) {
            window.LCPFF.application.nextStep();
          }
        }
      });
    }
  });
  /**
   * Initialize compliance form functionality
   */
  function initComplianceForm() {
    const complianceItems = document.querySelectorAll('input[name="complianceItems"]');
    const declarationDate = document.getElementById('declaration-date');
    
    // Set today's date as default for declaration
    if (declarationDate) {
      const today = new Date().toISOString().split('T')[0];
      declarationDate.value = today;
    }

    // Handle compliance item changes
    complianceItems.forEach(function(checkbox) {
      checkbox.addEventListener('change', function() {
        updateComplianceStatus(checkbox);
      });
    });

    // Initialize file upload tracking for compliance items
    initComplianceFileTracking();
  }

  /**
   * Update compliance status based on checkbox and file upload
   */
  function updateComplianceStatus(checkbox) {
    const complianceValue = checkbox.value;
    const statusElement = document.getElementById('status-' + complianceValue);
    const complianceItem = checkbox.closest('.compliance-item');
    
    if (!statusElement || !complianceItem) return;

    if (checkbox.checked) {
      // Check if file is uploaded for this compliance item
      const fileUploadArea = complianceItem.querySelector('.file-upload-area[data-compliance="' + complianceValue + '"]');
      const hasFile = fileUploadArea && fileUploadArea.classList.contains('success');
      
      if (hasFile) {
        statusElement.innerHTML = '<span class="status-completed">Completed</span>';
        complianceItem.classList.add('completed');
      } else {
        statusElement.innerHTML = '<span class="status-pending">File Required</span>';
        complianceItem.classList.remove('completed');
      }
    } else {
      if (complianceItem.classList.contains('optional')) {
        statusElement.innerHTML = '<span class="status-optional">Optional</span>';
      } else {
        statusElement.innerHTML = '<span class="status-pending">Pending Upload</span>';
      }
      complianceItem.classList.remove('completed');
    }
  }

  /**
   * Initialize compliance file upload tracking
   */
  function initComplianceFileTracking() {
    const complianceUploads = document.querySelectorAll('.file-upload-area[data-compliance]');
    
    complianceUploads.forEach(function(uploadArea) {
      const complianceValue = uploadArea.getAttribute('data-compliance');
      const fileInput = uploadArea.querySelector('.file-input');
      
      if (!fileInput) return;

      // Override the file selection handler to update compliance status
      fileInput.addEventListener('change', function() {
        setTimeout(function() {
          // Check if file was successfully uploaded
          if (uploadArea.classList.contains('success')) {
            const checkbox = document.getElementById(complianceValue);
            if (checkbox) {
              checkbox.checked = true;
              updateComplianceStatus(checkbox);
            }
          }
        }, 100);
      });

      // Handle file removal
      const observer = new MutationObserver(function(mutations) {
        mutations.forEach(function(mutation) {
          if (mutation.type === 'attributes' && mutation.attributeName === 'class') {
            if (!uploadArea.classList.contains('success')) {
              const checkbox = document.getElementById(complianceValue);
              if (checkbox) {
                updateComplianceStatus(checkbox);
              }
            }
          }
        });
      });

      observer.observe(uploadArea, { attributes: true });
    });
  }

  /**
   * Validate compliance form
   */
  function validateComplianceForm(form) {
    let isValid = true;
    
    // Standard field validation
    const requiredFields = form.querySelectorAll('[required]');
    requiredFields.forEach(function(field) {
      if (!validateField(field)) {
        isValid = false;
      }
    });
    
    // Compliance items validation
    const requiredComplianceItems = ['company-registration', 'tax-clearance', 'bccsa-compliance', 'local-content'];
    
    requiredComplianceItems.forEach(function(itemId) {
      const checkbox = document.getElementById(itemId);
      const complianceItem = checkbox ? checkbox.closest('.compliance-item') : null;
      
      if (checkbox && complianceItem) {
        const hasFile = complianceItem.querySelector('.file-upload-area.success');
        
        if (!checkbox.checked || !hasFile) {
          checkbox.classList.add('is-invalid');
          isValid = false;
          
          // Show error message
          showValidationMessage('Please complete all required compliance items with file uploads', 'error');
        } else {
          checkbox.classList.remove('is-invalid');
        }
      }
    });
    
    // Legal clearances validation
    const legalClearancesValid = validateCheckboxGroup('legalClearances');
    if (!legalClearancesValid) {
      isValid = false;
    }
    
    // Compliance declaration validation
    const declarationCheckbox = document.getElementById('compliance-declaration');
    if (declarationCheckbox && !declarationCheckbox.checked) {
      declarationCheckbox.classList.add('is-invalid');
      isValid = false;
    }
    
    return isValid;
  }

  /**
   * Get compliance progress
   */
  function getComplianceProgress() {
    const totalRequired = 4; // company-registration, tax-clearance, bccsa-compliance, local-content
    let completed = 0;
    
    const requiredItems = ['company-registration', 'tax-clearance', 'bccsa-compliance', 'local-content'];
    
    requiredItems.forEach(function(itemId) {
      const checkbox = document.getElementById(itemId);
      const complianceItem = checkbox ? checkbox.closest('.compliance-item') : null;
      
      if (checkbox && complianceItem && checkbox.checked) {
        const hasFile = complianceItem.querySelector('.file-upload-area.success');
        if (hasFile) {
          completed++;
        }
      }
    });
    
    return {
      completed: completed,
      total: totalRequired,
      percentage: Math.round((completed / totalRequired) * 100)
    };
  }

  // Initialize compliance form when DOM is loaded
  document.addEventListener('DOMContentLoaded', function() {
    initComplianceForm();
    
    // Override form validation for compliance form
    const complianceForm = document.getElementById('compliance-form');
    if (complianceForm) {
      complianceForm.addEventListener('submit', function(event) {
        event.preventDefault();
        
        if (validateComplianceForm(complianceForm)) {
          // Save form data and proceed to next step
          const formData = new FormData(complianceForm);
          const data = {};
          
          // Handle checkbox groups
          const complianceItems = [];
          const legalClearances = [];
          
          const checkboxes = complianceForm.querySelectorAll('input[type="checkbox"]:checked');
          checkboxes.forEach(function(checkbox) {
            if (checkbox.name === 'complianceItems') {
              complianceItems.push(checkbox.value);
            } else if (checkbox.name === 'legalClearances') {
              legalClearances.push(checkbox.value);
            }
          });
          
          // Add checkbox data
          if (complianceItems.length > 0) {
            data.complianceItems = complianceItems;
          }
          if (legalClearances.length > 0) {
            data.legalClearances = legalClearances;
          }
          
          // Add other form data
          for (let [key, value] of formData.entries()) {
            if (key !== 'complianceItems' && key !== 'legalClearances') {
              data[key] = value;
            }
          }
          
          // Add compliance progress
          data.complianceProgress = getComplianceProgress();
          
          // Save to localStorage
          const existingData = JSON.parse(storage.getItem('lcpff-application') || '{}');
          const updatedData = { ...existingData, ...data };
          storage.setItem('lcpff-application', JSON.stringify(updatedData));
          
          // Show save confirmation and proceed
          showSaveConfirmation();
          if (window.LCPFF && window.LCPFF.application) {
            window.LCPFF.application.nextStep();
          }
        }
      });
    }
  });
  /**
   * Initialize review and submit functionality
   */
  function initReviewSubmit() {
    const toggleButtons = document.querySelectorAll('.toggle-details');
    const editButtons = document.querySelectorAll('.edit-step');
    const submitButton = document.getElementById('submit-application');
    const saveDraftButton = document.getElementById('save-final-draft');
    
    // Initialize toggle functionality
    toggleButtons.forEach(function(button) {
      button.addEventListener('click', function() {
        const stepNumber = button.getAttribute('data-step');
        toggleStepDetails(stepNumber, button);
      });
    });

    // Initialize edit functionality
    editButtons.forEach(function(button) {
      button.addEventListener('click', function() {
        const stepNumber = parseInt(button.getAttribute('data-step'));
        editStep(stepNumber);
      });
    });

    // Initialize submit functionality
    if (submitButton) {
      submitButton.addEventListener('click', handleApplicationSubmission);
    }

    if (saveDraftButton) {
      saveDraftButton.addEventListener('click', function() {
        saveApplicationDraft();
        showSaveConfirmation();
      });
    }

    // Populate review data
    populateReviewData();
    
    // Initialize final checks validation
    initFinalChecksValidation();
  }

  /**
   * Toggle step details visibility
   */
  function toggleStepDetails(stepNumber, button) {
    const stepContent = document.querySelector('.review-step[data-step="' + stepNumber + '"] .review-step-content');
    const toggleText = button.querySelector('.toggle-text');
    const toggleIcon = button.querySelector('.toggle-icon');
    
    if (!stepContent) return;

    if (stepContent.style.display === 'none' || !stepContent.style.display) {
      stepContent.style.display = 'block';
      toggleText.textContent = 'Hide Details';
      button.classList.add('expanded');
    } else {
      stepContent.style.display = 'none';
      toggleText.textContent = 'Show Details';
      button.classList.remove('expanded');
    }
  }

  /**
   * Edit specific step
   */
  function editStep(stepNumber) {
    if (window.LCPFF && window.LCPFF.application) {
      window.LCPFF.application.navigateToStep(stepNumber);
    }
  }

  /**
   * Populate review data from localStorage
   */
  function populateReviewData() {
    const savedData = JSON.parse(storage.getItem('lcpff-application') || '{}');
    
    // Populate Step 1 - Company Information
    populateStepReview(1, {
      'Company Name': savedData.companyName || '',
      'Registration Number': savedData.registrationNumber || '',
      'Company Type': savedData.companyType || '',
      'Years in Operation': savedData.yearsOperation || '',
      'Contact Person': savedData.contactPerson || '',
      'Email': savedData.contactEmail || '',
      'Phone': savedData.contactPhone || '',
      'Address': [savedData.companyAddress, savedData.companyCity, savedData.companyProvince].filter(Boolean).join(', ') || ''
    });

    // Populate Step 2 - Project Overview
    populateStepReview(2, {
      'Project Title': savedData.projectTitle || '',
      'Logline': savedData.projectLogline || '',
      'Primary Genre': savedData.projectGenre || '',
      'Format': savedData.projectFormat || '',
      'Duration': savedData.projectDuration || '',
      'Development Stage': savedData.developmentStage || '',
      'Primary Language': savedData.primaryLanguage || '',
      'Target Audience': Array.isArray(savedData.targetAudience) ? savedData.targetAudience.join(', ') : ''
    });

    // Populate Step 3 - Creative Materials
    populateStepReview(3, {
      'Script Stage': savedData.scriptStage || '',
      'Visual Style': savedData.visualDescription || '',
      'Materials Notes': savedData.materialsNotes || ''
    });

    // Populate Step 4 - Financial Information
    populateStepReview(4, {
      'Total Budget': savedData.totalBudget ? 'R' + formatCurrency(savedData.totalBudget) : '',
      'Funding Requested': savedData.fundingRequested ? 'R' + formatCurrency(savedData.fundingRequested) : '',
      'Funding Percentage': savedData.fundingPercentage ? savedData.fundingPercentage + '%' : '',
      'Co-financing': savedData.coFinancing ? 'R' + formatCurrency(savedData.coFinancing) : '',
      'Cash Flow Needs': savedData.cashFlowNeeds || '',
      'Financing Sources': Array.isArray(savedData.financingSources) ? savedData.financingSources.join(', ') : ''
    });

    // Populate Step 5 - Compliance Documentation
    const complianceProgress = savedData.complianceProgress || { completed: 0, total: 4, percentage: 0 };
    populateStepReview(5, {
      'Compliance Items': Array.isArray(savedData.complianceItems) ? savedData.complianceItems.length + ' items completed' : '0 items completed',
      'Legal Clearances': Array.isArray(savedData.legalClearances) ? savedData.legalClearances.join(', ') : '',
      'Compliance Progress': complianceProgress.completed + '/' + complianceProgress.total + ' (' + complianceProgress.percentage + '%)',
      'Declarant': savedData.declarantName || '',
      'Declaration Date': savedData.declarationDate || ''
    });
  }

  /**
   * Populate individual step review
   */
  function populateStepReview(stepNumber, data) {
    const reviewContainer = document.getElementById('review-step-' + stepNumber);
    if (!reviewContainer) return;

    reviewContainer.innerHTML = '';

    Object.keys(data).forEach(function(label) {
      const value = data[label];
      const fieldDiv = document.createElement('div');
      fieldDiv.className = 'review-field';
      
      fieldDiv.innerHTML = `
        <div class="review-field-label">${label}</div>
        <div class="review-field-value ${value ? '' : 'empty'}">${value || 'Not provided'}</div>
      `;
      
      reviewContainer.appendChild(fieldDiv);
    });
  }

  /**
   * Initialize final checks validation
   */
  function initFinalChecksValidation() {
    const finalChecks = document.querySelectorAll('input[name="finalChecks"]');
    const submitButton = document.getElementById('submit-application');
    
    function validateFinalChecks() {
      const allChecked = Array.from(finalChecks).every(checkbox => checkbox.checked);
      
      if (submitButton) {
        submitButton.disabled = !allChecked;
        if (allChecked) {
          submitButton.classList.remove('disabled');
        } else {
          submitButton.classList.add('disabled');
        }
      }
      
      return allChecked;
    }

    finalChecks.forEach(function(checkbox) {
      checkbox.addEventListener('change', validateFinalChecks);
    });

    // Initial validation
    validateFinalChecks();
  }

  /**
   * Handle application submission
   */
  function handleApplicationSubmission() {
    const finalChecks = document.querySelectorAll('input[name="finalChecks"]');
    const allChecked = Array.from(finalChecks).every(checkbox => checkbox.checked);
    
    if (!allChecked) {
      showValidationMessage('Please complete all final checks before submitting', 'error');
      return;
    }

    // Show confirmation dialog
    if (confirm('Are you sure you want to submit your application? This action cannot be undone.')) {
      submitApplication();
    }
  }

  /**
   * Submit application
   */
  function submitApplication() {
    const submitButton = document.getElementById('submit-application');
    const originalText = submitButton.innerHTML;
    
    // Show loading state
    submitButton.disabled = true;
    submitButton.innerHTML = `
      <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor" style="margin-right: 8px; animation: spin 1s linear infinite;">
        <path d="M12,4V2A10,10 0 0,0 2,12H4A8,8 0 0,1 12,4Z"/>
      </svg>
      Submitting...
    `;

    // Add spinning animation
    const style = document.createElement('style');
    style.textContent = `
      @keyframes spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
      }
    `;
    document.head.appendChild(style);

    // Simulate submission process
    setTimeout(function() {
      // Generate application reference number
      const referenceNumber = 'LCPFF-' + new Date().getFullYear() + '-' + String(Math.floor(Math.random() * 10000)).padStart(4, '0');
      
      // Save submission data
      const submissionData = {
        referenceNumber: referenceNumber,
        submissionDate: new Date().toISOString(),
        status: 'submitted',
        submittedData: JSON.parse(storage.getItem('lcpff-application') || '{}')
      };
      
      storage.setItem('lcpff-application-submission', JSON.stringify(submissionData));
      
      // Show success message
      showSubmissionSuccess(referenceNumber);
      
      // Reset button
      submitButton.disabled = false;
      submitButton.innerHTML = originalText;
      
      // Remove style
      document.head.removeChild(style);
      
    }, 3000);
  }

  /**
   * Show submission success
   */
  function showSubmissionSuccess(referenceNumber) {
    // Create success modal
    const modal = document.createElement('div');
    modal.className = 'submission-success-modal';
    modal.style.cssText = `
      position: fixed;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      background-color: rgba(0, 0, 0, 0.8);
      display: flex;
      align-items: center;
      justify-content: center;
      z-index: 10000;
      padding: 20px;
    `;
    
    modal.innerHTML = `
      <div style="
        background-color: white;
        border-radius: 12px;
        padding: 40px;
        max-width: 500px;
        width: 100%;
        text-align: center;
        box-shadow: 0 20px 40px rgba(0, 0, 0, 0.3);
      ">
        <div style="
          width: 80px;
          height: 80px;
          background-color: var(--color-success);
          border-radius: 50%;
          display: flex;
          align-items: center;
          justify-content: center;
          margin: 0 auto 24px;
        ">
          <svg width="40" height="40" viewBox="0 0 24 24" fill="white">
            <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z"/>
          </svg>
        </div>
        <h2 style="
          font-family: var(--font-heading);
          font-size: 28px;
          color: var(--color-primary-black);
          margin-bottom: 16px;
          font-weight: 600;
        ">Application Submitted Successfully!</h2>
        <p style="
          color: var(--color-gray-medium);
          margin-bottom: 24px;
          line-height: 1.6;
        ">Your application has been submitted and assigned reference number:</p>
        <div style="
          background-color: var(--color-gray-lightest);
          padding: 16px;
          border-radius: 8px;
          margin-bottom: 24px;
          border: 2px solid var(--color-primary-gold);
        ">
          <strong style="
            font-family: var(--font-heading);
            font-size: 24px;
            color: var(--color-primary-black);
            letter-spacing: 1px;
          ">${referenceNumber}</strong>
        </div>
        <p style="
          color: var(--color-gray-medium);
          margin-bottom: 32px;
          font-size: 14px;
          line-height: 1.6;
        ">You will receive a confirmation email shortly. You can track your application progress in your dashboard.</p>
        <div style="display: flex; gap: 16px; justify-content: center;">
          <button onclick="window.location.href='dashboard.html'" style="
            background-color: var(--color-primary-gold);
            color: var(--color-primary-black);
            border: none;
            padding: 12px 24px;
            border-radius: 6px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
          ">Go to Dashboard</button>
          <button onclick="this.closest('.submission-success-modal').remove()" style="
            background-color: transparent;
            color: var(--color-gray-medium);
            border: 2px solid var(--color-gray-lighter);
            padding: 12px 24px;
            border-radius: 6px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.3s ease;
          ">Close</button>
        </div>
      </div>
    `;
    
    document.body.appendChild(modal);
    
    // Auto-remove after 30 seconds
    setTimeout(function() {
      if (document.body.contains(modal)) {
        document.body.removeChild(modal);
      }
    }, 30000);
  }

  /**
   * Save application draft
   */
  function saveApplicationDraft() {
    const draftData = {
      lastSaved: new Date().toISOString(),
      status: 'draft',
      currentStep: 6,
      data: JSON.parse(storage.getItem('lcpff-application') || '{}')
    };
    
    storage.setItem('lcpff-application-draft', JSON.stringify(draftData));
  }

  // Initialize review and submit when DOM is loaded
  document.addEventListener('DOMContentLoaded', function() {
    // Only initialize if we're on step 6
    const step6 = document.getElementById('step-6');
    if (step6) {
      // Use a timeout to ensure other initialization is complete
      setTimeout(initReviewSubmit, 100);
    }
  });
/**
 * Dashboard Notifications Functionality
 */
function initDashboardNotifications() {
  const markAllReadBtn = document.getElementById('mark-all-read');
  const notificationsList = document.getElementById('notifications-list');
  const dismissButtons = document.querySelectorAll('.notification-dismiss');

  if (!notificationsList) return;

  // Initialize notification state from localStorage
  loadNotificationState();

  // Mark all notifications as read
  if (markAllReadBtn) {
    markAllReadBtn.addEventListener('click', function() {
      markAllNotificationsRead();
    });
  }

  // Handle individual notification dismissal
  dismissButtons.forEach(function(button) {
    button.addEventListener('click', function() {
      const notificationId = button.getAttribute('data-notification-id');
      dismissNotification(notificationId);
    });
  });

  // Handle notification click to mark as read
  const notificationItems = document.querySelectorAll('.notification-item');
  notificationItems.forEach(function(item) {
    item.addEventListener('click', function(event) {
      // Don't mark as read if clicking on action buttons
      if (event.target.classList.contains('notification-action-btn') || 
          event.target.classList.contains('notification-dismiss')) {
        return;
      }
      
      const notificationId = item.getAttribute('data-notification-id');
      markNotificationRead(notificationId);
    });
  });
}

/**
 * Load notification state from localStorage
 */
function loadNotificationState() {
  const notificationState = JSON.parse(storage.getItem('lcpff-notifications') || '{}');
  
  Object.keys(notificationState).forEach(function(notificationId) {
    const state = notificationState[notificationId];
    const notificationElement = document.querySelector('[data-notification-id="' + notificationId + '"]');
    
    if (notificationElement) {
      if (state.read) {
        notificationElement.classList.remove('unread');
        notificationElement.classList.add('read');
      }
      
      if (state.dismissed) {
        notificationElement.style.display = 'none';
      }
    }
  });
  
  updateNotificationCounts();
}

/**
 * Save notification state to localStorage
 */
function saveNotificationState() {
  const notificationState = {};
  const notificationItems = document.querySelectorAll('.notification-item');
  
  notificationItems.forEach(function(item) {
    const notificationId = item.getAttribute('data-notification-id');
    notificationState[notificationId] = {
      read: item.classList.contains('read'),
      dismissed: item.style.display === 'none'
    };
  });
  
  storage.setItem('lcpff-notifications', JSON.stringify(notificationState));
}

/**
 * Mark all notifications as read
 */
function markAllNotificationsRead() {
  const unreadNotifications = document.querySelectorAll('.notification-item.unread');
  
  unreadNotifications.forEach(function(notification) {
    notification.classList.remove('unread');
    notification.classList.add('read');
  });
  
  saveNotificationState();
  updateNotificationCounts();
  
  // Show confirmation
  showNotificationFeedback('All notifications marked as read');
}

/**
 * Mark individual notification as read
 */
function markNotificationRead(notificationId) {
  const notification = document.querySelector('[data-notification-id="' + notificationId + '"]');
  
  if (notification && notification.classList.contains('unread')) {
    notification.classList.remove('unread');
    notification.classList.add('read');
    
    saveNotificationState();
    updateNotificationCounts();
  }
}

/**
 * Dismiss individual notification
 */
function dismissNotification(notificationId) {
  const notification = document.querySelector('[data-notification-id="' + notificationId + '"]');
  
  if (notification) {
    notification.style.opacity = '0';
    notification.style.transform = 'translateX(100%)';
    
    setTimeout(function() {
      notification.style.display = 'none';
      saveNotificationState();
      updateNotificationCounts();
      
      // Check if all notifications are dismissed
      const visibleNotifications = document.querySelectorAll('.notification-item[style*="display: none"]');
      const totalNotifications = document.querySelectorAll('.notification-item');
      
      if (visibleNotifications.length === totalNotifications.length) {
        showEmptyNotificationsState();
      }
    }, 300);
  }
}

/**
 * Update notification counts and badges
 */
function updateNotificationCounts() {
  const unreadCount = document.querySelectorAll('.notification-item.unread[style*="display: none"]').length;
  const totalUnread = document.querySelectorAll('.notification-item.unread').length - unreadCount;
  
  // Update any notification badges or counters
  const notificationBadges = document.querySelectorAll('.notification-badge');
  notificationBadges.forEach(function(badge) {
    if (totalUnread > 0) {
      badge.textContent = totalUnread;
      badge.style.display = 'inline-block';
    } else {
      badge.style.display = 'none';
    }
  });
  
  // Update mark all read button state
  const markAllReadBtn = document.getElementById('mark-all-read');
  if (markAllReadBtn) {
    markAllReadBtn.disabled = totalUnread === 0;
  }
}

/**
 * Show empty notifications state
 */
function showEmptyNotificationsState() {
  const notificationsList = document.getElementById('notifications-list');
  
  if (notificationsList) {
    notificationsList.innerHTML = `
      <div class="empty-notifications">
        <div class="empty-icon">
          <svg width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1">
            <path d="M18 8A6 6 0 0 0 6 8c0 7-3 9-3 9h18s-3-2-3-9"></path>
            <path d="M13.73 21a2 2 0 0 1-3.46 0"></path>
          </svg>
        </div>
        <h4>All caught up!</h4>
        <p>You have no new notifications at this time.</p>
      </div>
    `;
  }
}

/**
 * Show notification feedback
 */
function showNotificationFeedback(message) {
  const feedback = document.createElement('div');
  feedback.className = 'notification-feedback';
  feedback.textContent = message;
  feedback.style.cssText = `
    position: fixed;
    top: 20px;
    right: 20px;
    background-color: var(--color-success);
    color: white;
    padding: 12px 20px;
    border-radius: 4px;
    z-index: 1000;
    font-size: 14px;
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
    transition: all 0.3s ease;
  `;
  
  document.body.appendChild(feedback);
  
  setTimeout(function() {
    feedback.style.opacity = '0';
    setTimeout(function() {
      if (document.body.contains(feedback)) {
        document.body.removeChild(feedback);
      }
    }, 300);
  }, 3000);
}

/**
 * Add new notification (for demo purposes)
 */
function addNotification(notification) {
  const notificationsList = document.getElementById('notifications-list');
  if (!notificationsList) return;
  
  // Check if empty state is showing
  const emptyState = notificationsList.querySelector('.empty-notifications');
  if (emptyState) {
    notificationsList.innerHTML = '';
  }
  
  const notificationElement = document.createElement('div');
  notificationElement.className = 'notification-item unread';
  notificationElement.setAttribute('data-notification-id', notification.id);
  
  notificationElement.innerHTML = `
    <div class="notification-indicator"></div>
    <div class="notification-content">
      <div class="notification-header">
        <h4>${notification.title}</h4>
        <span class="notification-time">${notification.time}</span>
      </div>
      <p>${notification.message}</p>
      <div class="notification-actions">
        <button class="notification-action-btn">${notification.actionText || 'View Details'}</button>
        <button class="notification-dismiss" data-notification-id="${notification.id}">Dismiss</button>
      </div>
    </div>
  `;
  
  // Add to top of list
  notificationsList.insertBefore(notificationElement, notificationsList.firstChild);
  
  // Add event listeners
  const dismissBtn = notificationElement.querySelector('.notification-dismiss');
  dismissBtn.addEventListener('click', function() {
    dismissNotification(notification.id);
  });
  
  notificationElement.addEventListener('click', function(event) {
    if (event.target.classList.contains('notification-action-btn') || 
        event.target.classList.contains('notification-dismiss')) {
      return;
    }
    markNotificationRead(notification.id);
  });
  
  updateNotificationCounts();
  
  // Show animation
  notificationElement.style.opacity = '0';
  notificationElement.style.transform = 'translateY(-20px)';
  
  setTimeout(function() {
    notificationElement.style.transition = 'all 0.3s ease';
    notificationElement.style.opacity = '1';
    notificationElement.style.transform = 'translateY(0)';
  }, 100);
}

// Initialize dashboard notifications when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
  initDashboardNotifications();
});

// Export notification functions for potential use
window.LCPFF = window.LCPFF || {};
window.LCPFF.notifications = {
  addNotification: addNotification,
  markAllRead: markAllNotificationsRead,
  markRead: markNotificationRead,
  dismiss: dismissNotification
};
/**
 * Document Management Functionality
 */
(function() {
  'use strict';

  // Prevent multiple initialization
  if (window.LCPFF && window.LCPFF.documentManagementInitialized) {
    return;
  }

  /**
   * Debounced document search
   */
  const debounceDocumentSearch = window.LCPFF.utils.debounce(function(event) {
    const searchTerm = event.target.value.toLowerCase();
    filterDocuments();
  }, 300);

function initDocumentManagement() {
  const searchInput = document.getElementById('document-search');
  const typeFilter = document.getElementById('document-type-filter');
  const statusFilter = document.getElementById('document-status-filter');
  const uploadBtn = document.getElementById('upload-document');
  const organizeBtn = document.getElementById('organize-documents');
  const documentList = document.getElementById('document-list');

  if (!documentList) return;

  // Initialize search functionality
  if (searchInput) {
    searchInput.addEventListener('input', debounceDocumentSearch);
  }

  // Initialize filter functionality
  if (typeFilter) {
    typeFilter.addEventListener('change', filterDocuments);
  }

  if (statusFilter) {
    statusFilter.addEventListener('change', filterDocuments);
  }

  // Initialize upload functionality
  if (uploadBtn) {
    uploadBtn.addEventListener('click', function() {
      showUploadModal();
    });
  }

  // Initialize organize functionality
  if (organizeBtn) {
    organizeBtn.addEventListener('click', function() {
      toggleOrganizeMode();
    });
  }

  // Initialize document action buttons
  initDocumentActions();
}

/**
 * Filter documents based on search and filters
 */
function filterDocuments() {
  const searchTerm = document.getElementById('document-search')?.value.toLowerCase() || '';
  const typeFilter = document.getElementById('document-type-filter')?.value || '';
  const statusFilter = document.getElementById('document-status-filter')?.value || '';
  const documentItems = document.querySelectorAll('.document-item');

  let visibleCount = 0;

  documentItems.forEach(function(item) {
    const title = item.querySelector('h4')?.textContent.toLowerCase() || '';
    const type = item.getAttribute('data-type') || '';
    const status = item.getAttribute('data-status') || '';

    const matchesSearch = !searchTerm || title.includes(searchTerm);
    const matchesType = !typeFilter || type === typeFilter;
    const matchesStatus = !statusFilter || status === statusFilter;

    if (matchesSearch && matchesType && matchesStatus) {
      item.style.display = 'flex';
      visibleCount++;
    } else {
      item.style.display = 'none';
    }
  });

  // Update document count
  updateDocumentCount(visibleCount);

  // Show empty state if no documents match
  if (visibleCount === 0) {
    showEmptyDocumentState();
  } else {
    hideEmptyDocumentState();
  }
}

/**
 * Update document count display
 */
function updateDocumentCount(count) {
  const documentCount = document.querySelector('.document-count');
  if (documentCount) {
    const totalCount = document.querySelectorAll('.document-item').length;
    if (count === totalCount) {
      documentCount.textContent = `${count} documents`;
    } else {
      documentCount.textContent = `${count} of ${totalCount} documents`;
    }
  }
}

/**
 * Show empty document state
 */
function showEmptyDocumentState() {
  const documentList = document.getElementById('document-list');
  let emptyState = documentList.querySelector('.empty-documents');

  if (!emptyState) {
    emptyState = document.createElement('div');
    emptyState.className = 'empty-documents';
    emptyState.innerHTML = `
      <div class="empty-icon">
        <svg width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1">
          <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path>
          <polyline points="14,2 14,8 20,8"></polyline>
        </svg>
      </div>
      <h4>No documents found</h4>
      <p>Try adjusting your search or filter criteria.</p>
      <button class="btn btn-outline" onclick="clearDocumentFilters()">Clear Filters</button>
    `;
    emptyState.style.cssText = `
      text-align: center;
      padding: var(--spacing-3xl) var(--spacing-lg);
      color: var(--color-gray-medium);
    `;
    documentList.appendChild(emptyState);
  }

  emptyState.style.display = 'block';
}

/**
 * Hide empty document state
 */
function hideEmptyDocumentState() {
  const emptyState = document.querySelector('.empty-documents');
  if (emptyState) {
    emptyState.style.display = 'none';
  }
}

/**
 * Clear document filters
 */
function clearDocumentFilters() {
  const searchInput = document.getElementById('document-search');
  const typeFilter = document.getElementById('document-type-filter');
  const statusFilter = document.getElementById('document-status-filter');

  if (searchInput) searchInput.value = '';
  if (typeFilter) typeFilter.value = '';
  if (statusFilter) statusFilter.value = '';

  filterDocuments();
}

/**
 * Initialize document action buttons
 */
function initDocumentActions() {
  const documentItems = document.querySelectorAll('.document-item');

  documentItems.forEach(function(item) {
    const actionButtons = item.querySelectorAll('.document-action-btn');

    actionButtons.forEach(function(button) {
      button.addEventListener('click', function(event) {
        event.stopPropagation();
        const action = button.getAttribute('title').toLowerCase();
        const documentTitle = item.querySelector('h4')?.textContent || 'document';

        handleDocumentAction(action, documentTitle, item);
      });
    });
  });
}

/**
 * Handle document actions
 */
function handleDocumentAction(action, documentTitle, documentItem) {
  switch (action) {
    case 'download':
      downloadDocument(documentTitle);
      break;
    case 'preview':
      previewDocument(documentTitle);
      break;
    case 'delete':
      deleteDocument(documentTitle, documentItem);
      break;
    case 'replace':
      replaceDocument(documentTitle);
      break;
    case 'view comments':
      viewDocumentComments(documentTitle);
      break;
    default:
      console.log('Unknown action:', action);
  }
}

/**
 * Download document (mock)
 */
function downloadDocument(documentTitle) {
  showDocumentFeedback(`Downloading "${documentTitle}"...`);
  
  // Mock download delay
  setTimeout(function() {
    showDocumentFeedback(`"${documentTitle}" downloaded successfully`);
  }, 1500);
}

/**
 * Preview document (mock)
 */
function previewDocument(documentTitle) {
  showDocumentFeedback(`Opening preview for "${documentTitle}"...`);
  
  // Mock preview modal
  setTimeout(function() {
    alert(`Preview for "${documentTitle}" would open in a modal or new window.`);
  }, 500);
}

/**
 * Delete document (mock)
 */
function deleteDocument(documentTitle, documentItem) {
  if (confirm(`Are you sure you want to delete "${documentTitle}"?`)) {
    documentItem.style.opacity = '0';
    documentItem.style.transform = 'translateX(-100%)';
    
    setTimeout(function() {
      documentItem.remove();
      filterDocuments(); // Update counts
      showDocumentFeedback(`"${documentTitle}" deleted successfully`);
    }, 300);
  }
}

/**
 * Replace document (mock)
 */
function replaceDocument(documentTitle) {
  showDocumentFeedback(`Opening file picker to replace "${documentTitle}"...`);
  
  // Mock file picker
  setTimeout(function() {
    alert(`File picker for replacing "${documentTitle}" would open here.`);
  }, 500);
}

/**
 * View document comments (mock)
 */
function viewDocumentComments(documentTitle) {
  showDocumentFeedback(`Loading comments for "${documentTitle}"...`);
  
  // Mock comments modal
  setTimeout(function() {
    alert(`Comments for "${documentTitle}":\n\n• Please update the budget breakdown to include more detail on post-production costs.\n• The timeline seems optimistic - consider adding buffer time.\n• Overall structure looks good.`);
  }, 500);
}

/**
 * Show upload modal (mock)
 */
function showUploadModal() {
  alert('Document upload modal would open here with drag-and-drop functionality.');
}

/**
 * Toggle organize mode (mock)
 */
function toggleOrganizeMode() {
  const organizeBtn = document.getElementById('organize-documents');
  const documentItems = document.querySelectorAll('.document-item');
  
  if (organizeBtn.textContent === 'Organize') {
    // Enter organize mode
    organizeBtn.textContent = 'Done';
    organizeBtn.classList.add('active');
    
    documentItems.forEach(function(item) {
      item.classList.add('organize-mode');
      
      // Add drag handle
      const dragHandle = document.createElement('div');
      dragHandle.className = 'drag-handle';
      dragHandle.innerHTML = `
        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
          <line x1="3" y1="6" x2="21" y2="6"></line>
          <line x1="3" y1="12" x2="21" y2="12"></line>
          <line x1="3" y1="18" x2="21" y2="18"></line>
        </svg>
      `;
      item.insertBefore(dragHandle, item.firstChild);
    });
    
    showDocumentFeedback('Organize mode enabled - drag documents to reorder');
  } else {
    // Exit organize mode
    organizeBtn.textContent = 'Organize';
    organizeBtn.classList.remove('active');
    
    documentItems.forEach(function(item) {
      item.classList.remove('organize-mode');
      const dragHandle = item.querySelector('.drag-handle');
      if (dragHandle) {
        dragHandle.remove();
      }
    });
    
    showDocumentFeedback('Document organization saved');
  }
}

/**
 * Show document feedback
 */
function showDocumentFeedback(message) {
  const feedback = document.createElement('div');
  feedback.className = 'document-feedback';
  feedback.textContent = message;
  feedback.style.cssText = `
    position: fixed;
    top: 20px;
    right: 20px;
    background-color: var(--color-primary-gold);
    color: white;
    padding: 12px 20px;
    border-radius: 4px;
    z-index: 1000;
    font-size: 14px;
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
    transition: all 0.3s ease;
    max-width: 300px;
    word-wrap: break-word;
  `;
  
  document.body.appendChild(feedback);
  
  setTimeout(function() {
    feedback.style.opacity = '0';
    setTimeout(function() {
      if (document.body.contains(feedback)) {
        document.body.removeChild(feedback);
      }
    }, 300);
  }, 3000);
}

// Initialize document management when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
  initDocumentManagement();
});

// Export document management functions
window.LCPFF = window.LCPFF || {};
window.LCPFF.documents = {
  filter: filterDocuments,
  clearFilters: clearDocumentFilters,
  download: downloadDocument,
  preview: previewDocument,
  delete: deleteDocument
};

// Mark as initialized
window.LCPFF.documentManagementInitialized = true;

})(); // End of document management IIFE
/**
 * User Profile Management Functionality
 */
function initUserProfile() {
  const editProfileBtn = document.getElementById('edit-profile');
  const cancelEditBtn = document.getElementById('cancel-edit');
  const profileForm = document.getElementById('profile-form');
  const profileContent = document.getElementById('profile-content');
  const profileEditForm = document.getElementById('profile-edit-form');
  const changeAvatarBtn = document.getElementById('change-avatar');
  const preferenceCheckboxes = document.querySelectorAll('.preference-checkbox');

  if (!editProfileBtn || !profileContent || !profileEditForm) return;

  // Initialize edit profile functionality
  editProfileBtn.addEventListener('click', function() {
    showEditForm();
  });

  if (cancelEditBtn) {
    cancelEditBtn.addEventListener('click', function() {
      hideEditForm();
    });
  }

  // Initialize profile form submission
  if (profileForm) {
    profileForm.addEventListener('submit', function(event) {
      event.preventDefault();
      handleProfileUpdate();
    });
  }

  // Initialize avatar change functionality
  if (changeAvatarBtn) {
    changeAvatarBtn.addEventListener('click', function() {
      handleAvatarChange();
    });
  }

  // Initialize preference change tracking
  preferenceCheckboxes.forEach(function(checkbox) {
    checkbox.addEventListener('change', function() {
      handlePreferenceChange(checkbox);
    });
  });

  // Load saved preferences
  loadUserPreferences();
}

/**
 * Show profile edit form
 */
function showEditForm() {
  const profileContent = document.getElementById('profile-content');
  const profileEditForm = document.getElementById('profile-edit-form');
  const editProfileBtn = document.getElementById('edit-profile');

  if (profileContent && profileEditForm && editProfileBtn) {
    profileContent.style.display = 'none';
    profileEditForm.style.display = 'block';
    editProfileBtn.textContent = 'Cancel';
    editProfileBtn.classList.add('editing');
  }
}

/**
 * Hide profile edit form
 */
function hideEditForm() {
  const profileContent = document.getElementById('profile-content');
  const profileEditForm = document.getElementById('profile-edit-form');
  const editProfileBtn = document.getElementById('edit-profile');

  if (profileContent && profileEditForm && editProfileBtn) {
    profileContent.style.display = 'flex';
    profileEditForm.style.display = 'none';
    editProfileBtn.textContent = 'Edit Profile';
    editProfileBtn.classList.remove('editing');
  }
}

/**
 * Handle profile update
 */
function handleProfileUpdate() {
  const formData = new FormData(document.getElementById('profile-form'));
  const profileData = {};

  // Collect form data
  for (let [key, value] of formData.entries()) {
    profileData[key] = value;
  }

  // Validate form
  if (!validateProfileForm()) {
    return;
  }

  // Show loading state
  const submitBtn = document.querySelector('#profile-form button[type="submit"]');
  const originalText = submitBtn.textContent;
  submitBtn.disabled = true;
  submitBtn.textContent = 'Saving...';

  // Mock API call
  setTimeout(function() {
    // Update profile display
    updateProfileDisplay(profileData);
    
    // Save to localStorage
    storage.setItem('lcpff-profile', JSON.stringify(profileData));
    
    // Reset button and hide form
    submitBtn.disabled = false;
    submitBtn.textContent = originalText;
    hideEditForm();
    
    // Show success message
    showProfileFeedback('Profile updated successfully');
  }, 1500);
}

/**
 * Validate profile form
 */
function validateProfileForm() {
  const form = document.getElementById('profile-form');
  const requiredFields = form.querySelectorAll('[required]');
  let isValid = true;

  requiredFields.forEach(function(field) {
    if (!field.value.trim()) {
      field.classList.add('is-invalid');
      isValid = false;
    } else {
      field.classList.remove('is-invalid');
      field.classList.add('is-valid');
    }
  });

  // Email validation
  const emailField = document.getElementById('edit-email');
  if (emailField && emailField.value) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(emailField.value)) {
      emailField.classList.add('is-invalid');
      isValid = false;
    }
  }

  return isValid;
}

/**
 * Update profile display with new data
 */
function updateProfileDisplay(profileData) {
  const profileFields = {
    'edit-first-name': 'Full Name',
    'edit-last-name': 'Full Name',
    'edit-email': 'Email Address',
    'edit-phone': 'Phone Number',
    'edit-company': 'Company',
    'edit-role': 'Role'
  };

  // Update full name (combine first and last name)
  const firstName = document.getElementById('edit-first-name')?.value || '';
  const lastName = document.getElementById('edit-last-name')?.value || '';
  const fullNameField = Array.from(document.querySelectorAll('.profile-field')).find(
    field => field.querySelector('label')?.textContent === 'Full Name'
  );
  if (fullNameField) {
    const valueSpan = fullNameField.querySelector('.profile-value');
    if (valueSpan) {
      valueSpan.textContent = `${firstName} ${lastName}`.trim();
    }
  }

  // Update other fields
  Object.keys(profileFields).forEach(function(fieldId) {
    if (fieldId === 'edit-first-name' || fieldId === 'edit-last-name') return; // Already handled above
    
    const input = document.getElementById(fieldId);
    const labelText = profileFields[fieldId];
    
    if (input) {
      const profileField = Array.from(document.querySelectorAll('.profile-field')).find(
        field => field.querySelector('label')?.textContent === labelText
      );
      
      if (profileField) {
        const valueSpan = profileField.querySelector('.profile-value');
        if (valueSpan) {
          if (fieldId === 'edit-role') {
            // Capitalize role
            valueSpan.textContent = input.value.charAt(0).toUpperCase() + input.value.slice(1);
          } else {
            valueSpan.textContent = input.value;
          }
        }
      }
    }
  });
}

/**
 * Handle avatar change
 */
function handleAvatarChange() {
  // Create file input
  const fileInput = document.createElement('input');
  fileInput.type = 'file';
  fileInput.accept = 'image/*';
  fileInput.style.display = 'none';
  
  fileInput.addEventListener('change', function(event) {
    const file = event.target.files[0];
    if (file) {
      // Validate file
      if (file.size > 5 * 1024 * 1024) { // 5MB limit
        showProfileFeedback('Image must be smaller than 5MB', 'error');
        return;
      }
      
      if (!file.type.startsWith('image/')) {
        showProfileFeedback('Please select a valid image file', 'error');
        return;
      }
      
      // Mock upload
      showProfileFeedback('Uploading avatar...');
      
      setTimeout(function() {
        showProfileFeedback('Avatar updated successfully');
        // In a real app, you would update the avatar display here
      }, 2000);
    }
  });
  
  document.body.appendChild(fileInput);
  fileInput.click();
  document.body.removeChild(fileInput);
}

/**
 * Handle preference changes
 */
function handlePreferenceChange(checkbox) {
  const preferenceName = checkbox.parentNode.querySelector('.preference-text')?.textContent || 'preference';
  const isEnabled = checkbox.checked;
  
  // Save preference
  saveUserPreference(checkbox.id || preferenceName, isEnabled);
  
  // Show feedback
  const action = isEnabled ? 'enabled' : 'disabled';
  showProfileFeedback(`${preferenceName} ${action}`);
}

/**
 * Save user preference to localStorage
 */
function saveUserPreference(key, value) {
  const preferences = JSON.parse(storage.getItem('lcpff-preferences') || '{}');
  preferences[key] = value;
  storage.setItem('lcpff-preferences', JSON.stringify(preferences));
}

/**
 * Load user preferences from localStorage
 */
function loadUserPreferences() {
  const preferences = JSON.parse(storage.getItem('lcpff-preferences') || '{}');
  
  Object.keys(preferences).forEach(function(key) {
    const checkbox = document.getElementById(key) || 
                    Array.from(document.querySelectorAll('.preference-checkbox')).find(
                      cb => cb.parentNode.querySelector('.preference-text')?.textContent === key
                    );
    
    if (checkbox) {
      checkbox.checked = preferences[key];
    }
  });
}

/**
 * Show profile feedback
 */
function showProfileFeedback(message, type = 'success') {
  const feedback = document.createElement('div');
  feedback.className = 'profile-feedback';
  feedback.textContent = message;
  
  let backgroundColor;
  switch (type) {
    case 'error':
      backgroundColor = 'var(--color-error)';
      break;
    case 'warning':
      backgroundColor = 'var(--color-warning)';
      break;
    case 'success':
    default:
      backgroundColor = 'var(--color-success)';
      break;
  }
  
  feedback.style.cssText = `
    position: fixed;
    top: 20px;
    right: 20px;
    background-color: ${backgroundColor};
    color: ${type === 'warning' ? 'var(--color-primary-black)' : 'white'};
    padding: 12px 20px;
    border-radius: 4px;
    z-index: 1000;
    font-size: 14px;
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
    transition: all 0.3s ease;
    max-width: 300px;
    word-wrap: break-word;
  `;
  
  document.body.appendChild(feedback);
  
  setTimeout(function() {
    feedback.style.opacity = '0';
    setTimeout(function() {
      if (document.body.contains(feedback)) {
        document.body.removeChild(feedback);
      }
    }, 300);
  }, 3000);
}

/**
 * Handle security actions (mock)
 */
function initSecurityActions() {
  const securityButtons = document.querySelectorAll('.account-security .btn');
  
  securityButtons.forEach(function(button) {
    button.addEventListener('click', function() {
      const action = button.textContent.toLowerCase();
      
      switch (action) {
        case 'change password':
          showProfileFeedback('Password change form would open here');
          break;
        case 'enable 2fa':
          showProfileFeedback('Two-factor authentication setup would begin here');
          break;
        case 'manage sessions':
          showProfileFeedback('Session management interface would open here');
          break;
        default:
          showProfileFeedback('Security action: ' + action);
      }
    });
  });
}

// Initialize user profile when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
  initUserProfile();
  initSecurityActions();
});

// Export profile functions
window.LCPFF = window.LCPFF || {};
window.LCPFF.profile = {
  showEditForm: showEditForm,
  hideEditForm: hideEditForm,
  updateProfile: handleProfileUpdate,
  changeAvatar: handleAvatarChange,
  savePreference: saveUserPreference
};
/**
 * Project Showcase Functionality
 */
function initProjectShowcase() {
  initFeaturedCarousel();
  initProjectGallery();
  initProjectModal();
}

/**
 * Initialize Featured Projects Carousel
 */
function initFeaturedCarousel() {
  const carousel = document.getElementById('featured-carousel');
  const slides = document.querySelectorAll('.carousel-slide');
  const prevBtn = document.getElementById('carousel-prev');
  const nextBtn = document.getElementById('carousel-next');
  const indicators = document.querySelectorAll('.indicator');
  
  if (!carousel || slides.length === 0) return;

  let currentSlide = 0;
  const totalSlides = slides.length;
  let autoplayInterval;

  // Update carousel display
  function updateCarousel() {
    // Update slides
    slides.forEach((slide, index) => {
      slide.classList.toggle('active', index === currentSlide);
    });

    // Update indicators
    indicators.forEach((indicator, index) => {
      indicator.classList.toggle('active', index === currentSlide);
    });

    // Update navigation buttons
    if (prevBtn) prevBtn.disabled = currentSlide === 0;
    if (nextBtn) nextBtn.disabled = currentSlide === totalSlides - 1;
  }

  // Go to specific slide
  function goToSlide(slideIndex) {
    if (slideIndex >= 0 && slideIndex < totalSlides) {
      currentSlide = slideIndex;
      updateCarousel();
      resetAutoplay();
    }
  }

  // Next slide
  function nextSlide() {
    if (currentSlide < totalSlides - 1) {
      goToSlide(currentSlide + 1);
    } else {
      goToSlide(0); // Loop back to first slide
    }
  }

  // Previous slide
  function prevSlide() {
    if (currentSlide > 0) {
      goToSlide(currentSlide - 1);
    } else {
      goToSlide(totalSlides - 1); // Loop to last slide
    }
  }

  // Start autoplay
  function startAutoplay() {
    autoplayInterval = setInterval(nextSlide, 5000); // 5 seconds
  }

  // Stop autoplay
  function stopAutoplay() {
    if (autoplayInterval) {
      clearInterval(autoplayInterval);
      autoplayInterval = null;
    }
  }

  // Reset autoplay
  function resetAutoplay() {
    stopAutoplay();
    startAutoplay();
  }

  // Event listeners
  if (nextBtn) {
    nextBtn.addEventListener('click', nextSlide);
  }

  if (prevBtn) {
    prevBtn.addEventListener('click', prevSlide);
  }

  // Indicator clicks
  indicators.forEach((indicator, index) => {
    indicator.addEventListener('click', () => goToSlide(index));
  });

  // Keyboard navigation
  document.addEventListener('keydown', (event) => {
    if (event.key === 'ArrowLeft') {
      prevSlide();
    } else if (event.key === 'ArrowRight') {
      nextSlide();
    }
  });

  // Touch/swipe support
  let touchStartX = 0;
  let touchEndX = 0;

  carousel.addEventListener('touchstart', (event) => {
    touchStartX = event.changedTouches[0].screenX;
  });

  carousel.addEventListener('touchend', (event) => {
    touchEndX = event.changedTouches[0].screenX;
    handleSwipe();
  });

  function handleSwipe() {
    const swipeThreshold = 50;
    const diff = touchStartX - touchEndX;

    if (Math.abs(diff) > swipeThreshold) {
      if (diff > 0) {
        nextSlide(); // Swipe left - next slide
      } else {
        prevSlide(); // Swipe right - previous slide
      }
    }
  }

  // Pause autoplay on hover
  carousel.addEventListener('mouseenter', stopAutoplay);
  carousel.addEventListener('mouseleave', startAutoplay);

  // Initialize
  updateCarousel();
  startAutoplay();
}

/**
 * Initialize Project Gallery
 */
function initProjectGallery() {
  const projectGrid = document.getElementById('project-grid');
  const genreFilter = document.getElementById('genre-filter');
  const yearFilter = document.getElementById('year-filter');
  const fundingFilter = document.getElementById('funding-filter');
  const searchInput = document.getElementById('project-search');
  const loadMoreBtn = document.getElementById('load-more-projects');

  if (!projectGrid) return;

  // Mock project data
  const projects = [
    {
      id: 'amandla-rising',
      title: 'Amandla Rising',
      genre: 'drama',
      year: 2023,
      funding: 12000000,
      description: 'A powerful story of resilience during apartheid, following three generations of a family fighting for freedom.',
      thumbnail: "data:image/svg+xml,<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 400 300'><rect width='400' height='300' fill='%23D4AF37'/><text x='200' y='150' text-anchor='middle' dy='.3em' font-family='Arial' font-size='18' fill='%23000'>Amandla Rising</text></svg>",
      director: 'Jahmil X.T. Qubeka',
      cast: ['Thuso Mbedu', 'John Kani', 'Connie Ferguson']
    },
    {
      id: 'ubuntu-stories',
      title: 'Ubuntu Stories',
      genre: 'drama',
      year: 2024,
      funding: 15000000,
      description: 'A contemporary drama series exploring interconnected lives of diverse South African families.',
      thumbnail: "data:image/svg+xml,<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 400 300'><rect width='400' height='300' fill='%23000'/><text x='200' y='150' text-anchor='middle' dy='.3em' font-family='Arial' font-size='18' fill='%23D4AF37'>Ubuntu Stories</text></svg>",
      director: 'Nomsa Mbeki',
      cast: ['Various']
    },
    {
      id: 'voices-karoo',
      title: 'Voices of the Karoo',
      genre: 'documentary',
      year: 2023,
      funding: 3500000,
      description: 'An intimate documentary exploring the lives of people in the vast Karoo landscape.',
      thumbnail: "data:image/svg+xml,<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 400 300'><rect width='400' height='300' fill='%23D4AF37'/><text x='200' y='140' text-anchor='middle' dy='.3em' font-family='Arial' font-size='16' fill='%23000'>Voices of the</text><text x='200' y='160' text-anchor='middle' dy='.3em' font-family='Arial' font-size='16' fill='%23000'>Karoo</text></svg>",
      director: 'Sarah van der Merwe',
      cast: ['Documentary']
    },
    {
      id: 'township-tales',
      title: 'Township Tales',
      genre: 'comedy',
      year: 2022,
      funding: 8000000,
      description: 'A heartwarming comedy series about life, love, and laughter in a vibrant South African township.',
      thumbnail: "data:image/svg+xml,<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 400 300'><rect width='400' height='300' fill='%23000'/><text x='200' y='140' text-anchor='middle' dy='.3em' font-family='Arial' font-size='16' fill='%23D4AF37'>Township</text><text x='200' y='160' text-anchor='middle' dy='.3em' font-family='Arial' font-size='16' fill='%23D4AF37'>Tales</text></svg>",
      director: 'Kagiso Lediga',
      cast: ['Skeem Saam Cast']
    },
    {
      id: 'heritage-hunters',
      title: 'Heritage Hunters',
      genre: 'historical',
      year: 2021,
      funding: 20000000,
      description: 'An epic historical drama exploring South Africa\'s rich cultural heritage and ancient civilizations.',
      thumbnail: "data:image/svg+xml,<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 400 300'><rect width='400' height='300' fill='%23D4AF37'/><text x='200' y='140' text-anchor='middle' dy='.3em' font-family='Arial' font-size='16' fill='%23000'>Heritage</text><text x='200' y='160' text-anchor='middle' dy='.3em' font-family='Arial' font-size='16' fill='%23000'>Hunters</text></svg>",
      director: 'Gavin Hood',
      cast: ['International Cast']
    },
    {
      id: 'cape-noir',
      title: 'Cape Noir',
      genre: 'thriller',
      year: 2022,
      funding: 18000000,
      description: 'A gripping thriller set in Cape Town\'s underworld, exploring crime, corruption, and redemption.',
      thumbnail: "data:image/svg+xml,<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 400 300'><rect width='400' height='300' fill='%23000'/><text x='200' y='150' text-anchor='middle' dy='.3em' font-family='Arial' font-size='18' fill='%23D4AF37'>Cape Noir</text></svg>",
      director: 'Oliver Schmitz',
      cast: ['Presley Chweneyagae']
    }
  ];

  let filteredProjects = [...projects];
  let displayedProjects = 0;
  const projectsPerLoad = 6;

  // Format funding amount
  function formatFunding(amount) {
    if (amount >= 1000000) {
      return 'R' + (amount / 1000000).toFixed(1) + 'M';
    }
    return 'R' + amount.toLocaleString();
  }

  // Create project card HTML
  function createProjectCard(project) {
    return `
      <div class="project-card" data-project-id="${project.id}" onclick="openProjectModal('${project.id}')">
        <div class="project-thumbnail" style="background-image: url('${project.thumbnail}')"></div>
        <div class="project-info">
          <div class="project-meta">
            <span class="project-genre">${project.genre.charAt(0).toUpperCase() + project.genre.slice(1)}</span>
            <span class="project-year">${project.year}</span>
          </div>
          <h3 class="project-title">${project.title}</h3>
          <p class="project-description">${project.description}</p>
          <div class="project-funding">Funding: ${formatFunding(project.funding)}</div>
        </div>
      </div>
    `;
  }

  // Filter projects
  function filterProjects() {
    const genre = genreFilter ? genreFilter.value : '';
    const year = yearFilter ? yearFilter.value : '';
    const funding = fundingFilter ? fundingFilter.value : '';
    const search = searchInput ? searchInput.value.toLowerCase() : '';

    filteredProjects = projects.filter(project => {
      const matchesGenre = !genre || project.genre === genre;
      const matchesYear = !year || project.year.toString() === year;
      const matchesFunding = !funding || checkFundingRange(project.funding, funding);
      const matchesSearch = !search || 
        project.title.toLowerCase().includes(search) ||
        project.description.toLowerCase().includes(search) ||
        project.director.toLowerCase().includes(search);

      return matchesGenre && matchesYear && matchesFunding && matchesSearch;
    });

    displayedProjects = 0;
    projectGrid.innerHTML = '';
    loadMoreProjects();
  }

  // Check funding range
  function checkFundingRange(amount, range) {
    const amountInMillions = amount / 1000000;
    switch (range) {
      case '0-5':
        return amountInMillions <= 5;
      case '5-15':
        return amountInMillions > 5 && amountInMillions <= 15;
      case '15-30':
        return amountInMillions > 15 && amountInMillions <= 30;
      case '30+':
        return amountInMillions > 30;
      default:
        return true;
    }
  }

  // Load more projects
  function loadMoreProjects() {
    const projectsToShow = filteredProjects.slice(displayedProjects, displayedProjects + projectsPerLoad);
    
    projectsToShow.forEach(project => {
      projectGrid.innerHTML += createProjectCard(project);
    });

    displayedProjects += projectsToShow.length;

    // Update load more button
    if (loadMoreBtn) {
      if (displayedProjects >= filteredProjects.length) {
        loadMoreBtn.style.display = 'none';
      } else {
        loadMoreBtn.style.display = 'inline-flex';
        loadMoreBtn.textContent = `Load More Projects (${filteredProjects.length - displayedProjects} remaining)`;
      }
    }
  }

  // Event listeners
  if (genreFilter) {
    genreFilter.addEventListener('change', filterProjects);
  }

  if (yearFilter) {
    yearFilter.addEventListener('change', filterProjects);
  }

  if (fundingFilter) {
    fundingFilter.addEventListener('change', filterProjects);
  }

  if (searchInput) {
    searchInput.addEventListener('input', debounce(filterProjects, 300));
  }

  if (loadMoreBtn) {
    loadMoreBtn.addEventListener('click', loadMoreProjects);
  }

  // Initialize gallery
  filterProjects();

  // Store projects data globally for modal use
  window.LCPFF = window.LCPFF || {};
  window.LCPFF.projects = projects;
}

/**
 * Initialize Project Modal
 */
function initProjectModal() {
  // Create modal HTML if it doesn't exist
  if (!document.getElementById('project-modal')) {
    const modalHTML = `
      <div class="project-modal" id="project-modal">
        <div class="modal-content">
          <div class="modal-header" id="modal-header">
            <button class="modal-close" id="modal-close" aria-label="Close modal">
              <svg width="24" height="24" viewBox="0 0 24 24" fill="currentColor">
                <path d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z"/>
              </svg>
            </button>
          </div>
          <div class="modal-body" id="modal-body">
            <!-- Content will be populated by JavaScript -->
          </div>
        </div>
      </div>
    `;
    document.body.insertAdjacentHTML('beforeend', modalHTML);
  }

  const modal = document.getElementById('project-modal');
  const modalClose = document.getElementById('modal-close');

  // Close modal function
  function closeModal() {
    modal.classList.remove('active');
    document.body.style.overflow = '';
  }

  // Event listeners
  if (modalClose) {
    modalClose.addEventListener('click', closeModal);
  }

  // Close on backdrop click
  modal.addEventListener('click', (event) => {
    if (event.target === modal) {
      closeModal();
    }
  });

  // Close on escape key
  document.addEventListener('keydown', (event) => {
    if (event.key === 'Escape' && modal.classList.contains('active')) {
      closeModal();
    }
  });
}

/**
 * Open project modal with details
 */
function openProjectModal(projectId) {
  const modal = document.getElementById('project-modal');
  const modalHeader = document.getElementById('modal-header');
  const modalBody = document.getElementById('modal-body');
  
  if (!modal || !modalBody) return;

  // Get project data
  const projects = window.LCPFF && window.LCPFF.projects ? window.LCPFF.projects : [];
  const project = projects.find(p => p.id === projectId);

  if (!project) return;

  // Update modal header background
  modalHeader.style.backgroundImage = `url('${project.thumbnail}')`;

  // Create modal content
  const modalContent = `
    <h2 class="modal-title">${project.title}</h2>
    <div class="modal-meta">
      <span class="slide-genre">${project.genre.charAt(0).toUpperCase() + project.genre.slice(1)}</span>
      <span class="slide-year">${project.year}</span>
      <span class="slide-rating">★★★★☆</span>
    </div>
    <p class="modal-description">${project.description}</p>
    <div class="modal-details">
      <div class="modal-detail">
        <strong>Director:</strong>
        ${project.director}
      </div>
      <div class="modal-detail">
        <strong>Cast:</strong>
        ${project.cast.join(', ')}
      </div>
      <div class="modal-detail">
        <strong>Funding:</strong>
        ${formatFunding(project.funding)}
      </div>
      <div class="modal-detail">
        <strong>Status:</strong>
        ${project.year === 2024 ? 'In Production' : 'Completed'}
      </div>
    </div>
    <div class="modal-gallery">
      <h3>Gallery</h3>
      <div class="gallery-grid">
        <div class="gallery-item" style="background-image: url('${project.thumbnail}')"></div>
        <div class="gallery-item" style="background-image: url('${project.thumbnail}')"></div>
        <div class="gallery-item" style="background-image: url('${project.thumbnail}')"></div>
        <div class="gallery-item" style="background-image: url('${project.thumbnail}')"></div>
      </div>
    </div>
    <div class="modal-actions">
      <button class="btn btn-primary">Watch Trailer</button>
      <button class="btn btn-secondary">Download Press Kit</button>
      <button class="btn btn-tertiary">Share Project</button>
    </div>
  `;

  modalBody.innerHTML = modalContent;

  // Show modal
  modal.classList.add('active');
  document.body.style.overflow = 'hidden';

  // Helper function for formatting (defined locally if not available globally)
  function formatFunding(amount) {
    if (amount >= 1000000) {
      return 'R' + (amount / 1000000).toFixed(1) + 'M';
    }
    return 'R' + amount.toLocaleString();
  }
}

// Initialize project showcase when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
  initProjectShowcase();
});

// Make openProjectModal globally available
window.openProjectModal = openProjectModal;

  /**
   * Initialize resources page functionality
   */
  function initResourcesPage() {
    if (!document.querySelector('.resources-section')) return;

    initResourceFiltering();
    initResourceSearch();
    initFAQFunctionality();
  }

  /**
   * Initialize resource filtering functionality
   */
  function initResourceFiltering() {
    const categoryButtons = document.querySelectorAll('.category-btn');
    const filterSelect = document.getElementById('resource-filter');
    const resourceCards = document.querySelectorAll('.resource-card');

    // Category button filtering
    categoryButtons.forEach(function(button) {
      button.addEventListener('click', function() {
        const category = button.getAttribute('data-category');
        
        // Update active button
        categoryButtons.forEach(function(btn) {
          btn.classList.remove('active');
          btn.setAttribute('aria-pressed', 'false');
        });
        button.classList.add('active');
        button.setAttribute('aria-pressed', 'true');
        
        // Filter resources
        filterResources(category, filterSelect.value);
      });
    });

    // Type filter dropdown
    if (filterSelect) {
      filterSelect.addEventListener('change', function() {
        const activeCategory = document.querySelector('.category-btn.active').getAttribute('data-category');
        filterResources(activeCategory, filterSelect.value);
      });
    }
  }

  /**
   * Filter resources based on category and type
   */
  function filterResources(category, type) {
    const resourceCards = document.querySelectorAll('.resource-card');
    
    resourceCards.forEach(function(card) {
      const cardCategory = card.getAttribute('data-category');
      const cardType = card.getAttribute('data-type');
      
      let showCard = true;
      
      // Check category filter
      if (category !== 'all' && cardCategory !== category) {
        showCard = false;
      }
      
      // Check type filter
      if (type !== 'all' && cardType !== type) {
        showCard = false;
      }
      
      // Show/hide card with animation
      if (showCard) {
        card.classList.remove('hidden');
        card.style.display = 'flex';
      } else {
        card.classList.add('hidden');
        card.style.display = 'none';
      }
    });
    
    // Update results count
    updateResourcesCount();
  }

  /**
   * Initialize resource search functionality
   */
  function initResourceSearch() {
    const searchInput = document.getElementById('resource-search');
    
    if (!searchInput) return;
    
    let searchTimeout;
    
    searchInput.addEventListener('input', function() {
      clearTimeout(searchTimeout);
      searchTimeout = setTimeout(function() {
        performResourceSearch(searchInput.value.toLowerCase());
      }, 300);
    });
  }

  /**
   * Perform resource search
   */
  function performResourceSearch(searchTerm) {
    const resourceCards = document.querySelectorAll('.resource-card');
    const activeCategory = document.querySelector('.category-btn.active').getAttribute('data-category');
    const filterSelect = document.getElementById('resource-filter');
    const activeType = filterSelect ? filterSelect.value : 'all';
    
    resourceCards.forEach(function(card) {
      const title = card.querySelector('.resource-title').textContent.toLowerCase();
      const description = card.querySelector('.resource-description').textContent.toLowerCase();
      const cardCategory = card.getAttribute('data-category');
      const cardType = card.getAttribute('data-type');
      
      let showCard = true;
      
      // Check search term
      if (searchTerm && !title.includes(searchTerm) && !description.includes(searchTerm)) {
        showCard = false;
      }
      
      // Check category filter
      if (activeCategory !== 'all' && cardCategory !== activeCategory) {
        showCard = false;
      }
      
      // Check type filter
      if (activeType !== 'all' && cardType !== activeType) {
        showCard = false;
      }
      
      // Show/hide card
      if (showCard) {
        card.classList.remove('hidden');
        card.style.display = 'flex';
      } else {
        card.classList.add('hidden');
        card.style.display = 'none';
      }
    });
    
    updateResourcesCount();
  }

  /**
   * Update resources count display
   */
  function updateResourcesCount() {
    const visibleCards = document.querySelectorAll('.resource-card:not(.hidden)');
    const totalCards = document.querySelectorAll('.resource-card');
    
    // You could add a results counter here if needed
    console.log(`Showing ${visibleCards.length} of ${totalCards.length} resources`);
  }

  /**
   * Initialize FAQ functionality
   */
  function initFAQFunctionality() {
    initFAQAccordion();
    initFAQSearch();
    initFAQCategoryFiltering();
  }

  /**
   * Initialize FAQ accordion functionality
   */
  function initFAQAccordion() {
    const faqQuestions = document.querySelectorAll('.faq-question');
    
    faqQuestions.forEach(function(question) {
      question.addEventListener('click', function() {
        const faqItem = question.closest('.faq-item');
        const answer = faqItem.querySelector('.faq-answer');
        const isExpanded = question.getAttribute('aria-expanded') === 'true';
        
        // Close all other FAQ items
        faqQuestions.forEach(function(otherQuestion) {
          if (otherQuestion !== question) {
            const otherItem = otherQuestion.closest('.faq-item');
            const otherAnswer = otherItem.querySelector('.faq-answer');
            otherQuestion.setAttribute('aria-expanded', 'false');
            otherAnswer.classList.remove('active');
          }
        });
        
        // Toggle current FAQ item
        question.setAttribute('aria-expanded', !isExpanded);
        if (!isExpanded) {
          answer.classList.add('active');
        } else {
          answer.classList.remove('active');
        }
      });
      
      // Handle keyboard navigation
      question.addEventListener('keydown', function(event) {
        if (event.key === 'Enter' || event.key === ' ') {
          event.preventDefault();
          question.click();
        }
      });
    });
  }

  /**
   * Initialize FAQ search functionality
   */
  function initFAQSearch() {
    const faqSearchInput = document.getElementById('faq-search');
    
    if (!faqSearchInput) return;
    
    let searchTimeout;
    
    faqSearchInput.addEventListener('input', function() {
      clearTimeout(searchTimeout);
      searchTimeout = setTimeout(function() {
        performFAQSearch(faqSearchInput.value.toLowerCase());
      }, 300);
    });
  }

  /**
   * Perform FAQ search
   */
  function performFAQSearch(searchTerm) {
    const faqItems = document.querySelectorAll('.faq-item');
    const activeFAQCategory = document.querySelector('.faq-category-btn.active').getAttribute('data-faq-category');
    
    faqItems.forEach(function(item) {
      const question = item.querySelector('.faq-question span').textContent.toLowerCase();
      const answer = item.querySelector('.faq-answer').textContent.toLowerCase();
      const itemCategory = item.getAttribute('data-faq-category');
      
      let showItem = true;
      
      // Check search term
      if (searchTerm && !question.includes(searchTerm) && !answer.includes(searchTerm)) {
        showItem = false;
      }
      
      // Check category filter
      if (activeFAQCategory !== 'all' && itemCategory !== activeFAQCategory) {
        showItem = false;
      }
      
      // Show/hide item
      if (showItem) {
        item.classList.remove('hidden');
        item.style.display = 'block';
      } else {
        item.classList.add('hidden');
        item.style.display = 'none';
      }
    });
  }

  /**
   * Initialize FAQ category filtering
   */
  function initFAQCategoryFiltering() {
    const faqCategoryButtons = document.querySelectorAll('.faq-category-btn');
    
    faqCategoryButtons.forEach(function(button) {
      button.addEventListener('click', function() {
        const category = button.getAttribute('data-faq-category');
        
        // Update active button
        faqCategoryButtons.forEach(function(btn) {
          btn.classList.remove('active');
        });
        button.classList.add('active');
        
        // Filter FAQs
        filterFAQs(category);
      });
    });
  }

  /**
   * Filter FAQs based on category
   */
  function filterFAQs(category) {
    const faqItems = document.querySelectorAll('.faq-item');
    const faqSearchInput = document.getElementById('faq-search');
    const searchTerm = faqSearchInput ? faqSearchInput.value.toLowerCase() : '';
    
    faqItems.forEach(function(item) {
      const itemCategory = item.getAttribute('data-faq-category');
      const question = item.querySelector('.faq-question span').textContent.toLowerCase();
      const answer = item.querySelector('.faq-answer').textContent.toLowerCase();
      
      let showItem = true;
      
      // Check category filter
      if (category !== 'all' && itemCategory !== category) {
        showItem = false;
      }
      
      // Check search term
      if (searchTerm && !question.includes(searchTerm) && !answer.includes(searchTerm)) {
        showItem = false;
      }
      
      // Show/hide item
      if (showItem) {
        item.classList.remove('hidden');
        item.style.display = 'block';
      } else {
        item.classList.add('hidden');
        item.style.display = 'none';
      }
    });
  }

  // Global functions for resource actions (called from HTML)
  window.downloadResource = function(filename) {
    // Mock download functionality with analytics tracking
    console.log('Downloading resource:', filename);
    
    // Mock analytics tracking
    trackResourceDownload(filename);
    
    // Show download confirmation
    showDownloadConfirmation(filename);
  };

  window.previewResource = function(filename) {
    // Mock preview functionality with analytics tracking
    console.log('Previewing resource:', filename);
    
    // Mock analytics tracking
    trackResourcePreview(filename);
    
    // Show preview modal or new window
    showResourcePreview(filename);
  };

  window.playVideo = function(filename) {
    // Mock video play functionality with analytics tracking
    console.log('Playing video:', filename);
    
    // Mock analytics tracking
    trackVideoPlay(filename);
    
    // Show video player
    showVideoPlayer(filename);
  };

  /**
   * Track resource download for analytics
   */
  function trackResourceDownload(filename) {
    // Mock analytics event
    const analyticsData = {
      event: 'resource_download',
      filename: filename,
      timestamp: new Date().toISOString(),
      userAgent: navigator.userAgent,
      referrer: document.referrer
    };
    
    console.log('Analytics: Resource Download', analyticsData);
    
    // In a real implementation, this would send data to analytics service
    // Example: gtag('event', 'download', { 'file_name': filename });
    // Example: analytics.track('Resource Downloaded', analyticsData);
  }

  /**
   * Track resource preview for analytics
   */
  function trackResourcePreview(filename) {
    // Mock analytics event
    const analyticsData = {
      event: 'resource_preview',
      filename: filename,
      timestamp: new Date().toISOString(),
      userAgent: navigator.userAgent
    };
    
    console.log('Analytics: Resource Preview', analyticsData);
  }

  /**
   * Track video play for analytics
   */
  function trackVideoPlay(filename) {
    // Mock analytics event
    const analyticsData = {
      event: 'video_play',
      filename: filename,
      timestamp: new Date().toISOString(),
      userAgent: navigator.userAgent
    };
    
    console.log('Analytics: Video Play', analyticsData);
  }

  /**
   * Show download confirmation with file details
   */
  function showDownloadConfirmation(filename) {
    const fileInfo = getFileInfo(filename);
    const message = `Download started: ${fileInfo.displayName}\n\nFile Type: ${fileInfo.type}\nFile Size: ${fileInfo.size}\nLast Updated: ${fileInfo.updated}\n\nNote: This is a mockup - no actual file will be downloaded.`;
    alert(message);
  }

  /**
   * Show resource preview
   */
  function showResourcePreview(filename) {
    const fileInfo = getFileInfo(filename);
    const message = `Opening preview for: ${fileInfo.displayName}\n\nFile Type: ${fileInfo.type}\nFile Size: ${fileInfo.size}\n\nNote: This is a mockup - no actual preview will be shown.`;
    alert(message);
  }

  /**
   * Show video player
   */
  function showVideoPlayer(filename) {
    const fileInfo = getFileInfo(filename);
    const message = `Playing video: ${fileInfo.displayName}\n\nDuration: ${fileInfo.duration}\nFormat: ${fileInfo.format}\n\nNote: This is a mockup - no actual video will play.`;
    alert(message);
  }

  /**
   * Get file information for display
   */
  function getFileInfo(filename) {
    const fileDatabase = {
      'first-time-guide.pdf': {
        displayName: 'First-Time Applicant Guide',
        type: 'PDF Document',
        size: '2.4 MB',
        updated: 'January 2024',
        downloads: 1247
      },
      'financial-guide.pdf': {
        displayName: 'Financial Documentation Guide',
        type: 'PDF Document',
        size: '1.8 MB',
        updated: 'December 2023',
        downloads: 892
      },
      'application-walkthrough.mp4': {
        displayName: 'Application Process Walkthrough',
        type: 'Video Tutorial',
        duration: '45 minutes',
        format: 'MP4 (1080p)',
        updated: 'January 2024',
        views: 3456
      },
      'budget-template.xlsx': {
        displayName: 'Budget Breakdown Template',
        type: 'Excel Spreadsheet',
        size: '156 KB',
        updated: 'November 2023',
        downloads: 2134
      },
      'pitch-deck-template.pptx': {
        displayName: 'Project Pitch Deck Template',
        type: 'PowerPoint Presentation',
        size: '2.1 MB',
        updated: 'October 2023',
        downloads: 1678
      },
      'application-checklist.pdf': {
        displayName: 'Application Checklist',
        type: 'PDF Checklist',
        size: '245 KB',
        updated: 'January 2024',
        downloads: 3421
      }
    };
    
    return fileDatabase[filename] || {
      displayName: filename,
      type: 'Unknown',
      size: 'Unknown',
      updated: 'Unknown',
      downloads: 0
    };
  }

  // Update the DOMContentLoaded event listener to include resources page
  document.addEventListener('DOMContentLoaded', function() {
    initMobileNavigation();
    initAccessibilityFeatures();
    initSmoothScrolling();
    initFormValidation();
    initCounterAnimations();
    initResourcesPage(); // Add this line
  });

  /**
   * Initialize contact form functionality
   */
  function initContactForm() {
  const contactForm = document.querySelector('.contact-form');
  if (!contactForm) return;

  // Initialize character counter for message field
  const messageField = contactForm.querySelector('#contact-message');
  const characterCounter = contactForm.querySelector('.character-counter');
  
  if (messageField && characterCounter) {
    const currentCount = characterCounter.querySelector('.current-count');
    const maxLength = 1000;
    
    function updateCharacterCount() {
      const count = messageField.value.length;
      currentCount.textContent = count;
      
      if (count > maxLength) {
        characterCounter.classList.add('over-limit');
        messageField.classList.add('is-invalid');
      } else {
        characterCounter.classList.remove('over-limit');
        if (count >= 20) {
          messageField.classList.remove('is-invalid');
          messageField.classList.add('is-valid');
        }
      }
    }
    
    messageField.addEventListener('input', updateCharacterCount);
    updateCharacterCount(); // Initial count
  }

  // Enhanced validation for contact form
  contactForm.addEventListener('submit', function(event) {
    event.preventDefault();
    
    if (validateContactForm(contactForm)) {
      handleContactFormSubmission(contactForm);
    }
  });
}

/**
 * Validate contact form with specific rules
 */
function validateContactForm(form) {
  let isValid = true;
  
  // Standard field validation
  const requiredFields = form.querySelectorAll('[required]');
  requiredFields.forEach(function(field) {
    if (!validateField(field)) {
      isValid = false;
    }
  });
  
  // Message minimum length validation
  const messageField = form.querySelector('#contact-message');
  if (messageField && messageField.value.trim().length < 20) {
    messageField.classList.add('is-invalid');
    const feedback = messageField.parentNode.querySelector('.invalid-feedback');
    if (feedback) {
      feedback.textContent = 'Message must be at least 20 characters long.';
    }
    isValid = false;
  }
  
  // Message maximum length validation
  if (messageField && messageField.value.length > 1000) {
    messageField.classList.add('is-invalid');
    const feedback = messageField.parentNode.querySelector('.invalid-feedback');
    if (feedback) {
      feedback.textContent = 'Message must be less than 1000 characters.';
    }
    isValid = false;
  }
  
  return isValid;
}

/**
 * Handle contact form submission
 */
function handleContactFormSubmission(form) {
  const submitBtn = form.querySelector('button[type="submit"]');
  const originalText = submitBtn.textContent;
  
  // Show loading state
  submitBtn.disabled = true;
  submitBtn.textContent = 'Sending Message...';
  
  // Mock API call with realistic delay
  setTimeout(function() {
    // Reset button
    submitBtn.disabled = false;
    submitBtn.textContent = originalText;
    
    // Show success message
    showContactSuccessMessage();
    
    // Reset form
    form.reset();
    
    // Clear validation states
    const inputs = form.querySelectorAll('.form-control, .form-check-input');
    inputs.forEach(clearValidationState);
    
    // Reset character counter
    const characterCounter = form.querySelector('.character-counter .current-count');
    if (characterCounter) {
      characterCounter.textContent = '0';
    }
    
  }, 2500); // Slightly longer delay for contact form
}

/**
 * Show contact form success message
 */
function showContactSuccessMessage() {
  // Create success notification
  const notification = document.createElement('div');
  notification.className = 'contact-success-notification';
  notification.innerHTML = `
    <div class="notification-content">
      <svg width="24" height="24" viewBox="0 0 24 24" fill="currentColor">
        <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z"/>
      </svg>
      <div class="notification-text">
        <h4>Message Sent Successfully!</h4>
        <p>Thank you for contacting LCPFF. We'll get back to you within 24 hours.</p>
      </div>
    </div>
  `;
  
  notification.style.cssText = `
    position: fixed;
    top: 20px;
    right: 20px;
    background: linear-gradient(135deg, #28A745 0%, #20C997 100%);
    color: white;
    padding: 20px;
    border-radius: 8px;
    z-index: 1000;
    max-width: 400px;
    box-shadow: 0 8px 25px rgba(40, 167, 69, 0.3);
    transform: translateX(100%);
    transition: all 0.3s ease;
  `;
  
  // Add notification styles
  const style = document.createElement('style');
  style.textContent = `
    .contact-success-notification .notification-content {
      display: flex;
      align-items: flex-start;
      gap: 12px;
    }
    .contact-success-notification svg {
      flex-shrink: 0;
      margin-top: 2px;
    }
    .contact-success-notification h4 {
      margin: 0 0 4px 0;
      font-size: 16px;
      font-weight: 600;
    }
    .contact-success-notification p {
      margin: 0;
      font-size: 14px;
      opacity: 0.9;
    }
  `;
  document.head.appendChild(style);
  
  document.body.appendChild(notification);
  
  // Animate in
  setTimeout(function() {
    notification.style.transform = 'translateX(0)';
  }, 100);
  
  // Animate out and remove
  setTimeout(function() {
    notification.style.transform = 'translateX(100%)';
    setTimeout(function() {
      if (notification.parentNode) {
        document.body.removeChild(notification);
      }
      if (style.parentNode) {
        document.head.removeChild(style);
      }
    }, 300);
  }, 5000);
}

/**
 * Initialize animated counters for about page
 */
function initAboutPageCounters() {
  const impactCounters = document.querySelectorAll('.impact-number[data-target]');
  
  if (impactCounters.length === 0) return;

  const observerOptions = {
    threshold: 0.5,
    rootMargin: '0px 0px -100px 0px'
  };

  const observer = new IntersectionObserver(function(entries) {
    entries.forEach(function(entry) {
      if (entry.isIntersecting) {
        animateImpactCounter(entry.target);
        observer.unobserve(entry.target);
      }
    });
  }, observerOptions);

  impactCounters.forEach(function(counter) {
    observer.observe(counter);
  });
}

/**
 * Animate impact counter with proper formatting
 */
function animateImpactCounter(element) {
  const target = parseInt(element.getAttribute('data-target'));
  const duration = 2500; // Slightly longer for impact
  const increment = target / (duration / 16);
  let current = 0;

  const timer = setInterval(function() {
    current += increment;
    
    if (current >= target) {
      current = target;
      clearInterval(timer);
    }

    // Format the number based on its size and context
    let displayValue;
    if (target >= 1000000000) {
      // Billions (for funding amounts)
      displayValue = 'R' + (current / 1000000000).toFixed(1) + 'B';
    } else if (target >= 1000000) {
      // Millions
      displayValue = 'R' + (current / 1000000).toFixed(1) + 'M';
    } else if (target >= 1000) {
      // Thousands
      displayValue = (current / 1000).toFixed(0) + 'K';
    } else {
      // Regular numbers
      displayValue = Math.floor(current).toString();
    }

    // Handle percentage suffix
    const suffix = element.querySelector('.impact-suffix');
    if (suffix) {
      element.childNodes[0].textContent = Math.floor(current);
    } else {
      element.childNodes[0].textContent = displayValue;
    }
  }, 16);
}

// Initialize contact form and about page functionality when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
  initContactForm();
  initAboutPageCounters();
});

})();